#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library: disambiguation
 *   
 *   This module defines classes related to resolving ambiguity in noun
 *   phrases in command input.  
 */

#include "adv3.h"


/* ------------------------------------------------------------------------ */
/*
 *   Distinguisher.  This object encapsulates logic that determines
 *   whether or not we can tell two objects apart.
 *   
 *   Each game object has a list of distinguishers.  For most objects, the
 *   distinguisher list contains only BasicDistinguisher, since most game
 *   objects are unique and thus are inherently distinguishable from all
 *   other objects.  
 */
class Distinguisher: object
    /* can we distinguish the given two objects? */
    canDistinguish(a, b) { return true; }
;

/*
 *   "Basic" Distinguisher.  This distinguisher can tell two objects apart
 *   if one or the other object is not marked as isEquivalent, OR if the
 *   two objects don't have an identical superclass list.  This
 *   distinguisher thus can tell apart objects unless they're "basic
 *   equivalents," marked with isEquivalent and descended from the same
 *   base class or classes.  
 */
basicDistinguisher: Distinguisher
    canDistinguish(a, b)
    {
        /*
         *   If the two objects are both marked isEquivalent have have
         *   identical superclass lists, they are basic equivalents, so we
         *   cannot distinguish them.
         */
        return !(a.isEquivalent
                 && b.isEquivalent
                 && a.getSuperclassList() == b.getSuperclassList());
    }
;

/*
 *   Location/ownership Distinguisher.  This distinguisher can tell two
 *   objects apart if they have different owners, or are "unowned" and are
 *   in different immediate containers.  
 */
ownershipAndLocationDistinguisher: Distinguisher
    canDistinguish(a, b)
    {
        local aOwner;
        local bOwner;

        /* get the nominal owner of each object */
        aOwner = a.getNominalOwner();
        bOwner = b.getNominalOwner();

        /* 
         *   for each object, if it's directly in its nominal owner, use
         *   the nominal owner as the distinctive feature; otherwise, use
         *   the location with the owner 
         */
        if (aOwner != nil && !a.isDirectlyIn(aOwner))
            aOwner = nil;
        if (bOwner != nil && !b.isDirectlyIn(bOwner))
            bOwner = nil;

        /* 
         *   if neither object is owned, we can't tell them apart on the
         *   basis of ownership, so check to see if we can tell them apart
         *   on the basis of location 
         */
        if (aOwner == nil && bOwner == nil)
        {
            /* 
             *   neither is owned - we can tell them apart only if they
             *   have different immediate containers 
             */
            return a.location != b.location;
        }

        /*
         *   One or both objects are owned, so we can tell them apart if
         *   and only if they have different owners.  
         */
        return aOwner != bOwner;
    }
;

/*
 *   Lit/unlit Distinguisher.  This distinguisher can tell two objects
 *   apart if one is lit (i.e., its isLit property is true) and the other
 *   isn't. 
 */
litUnlitDistinguisher: Distinguisher
    canDistinguish(a, b)
    {
        /* we can tell them apart if one is lit and the other isn't */
        return a.isLit != b.isLit;
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   A command ranking criterion for comparing by the number of ordinal
 *   phrases ("first", "the second one") we find in a result. 
 */
rankByDisambigOrdinals: CommandRankingByProblem
    prop_ = &disambigOrdinalCount
;

/*
 *   Disambiguation Ranking.  This is a special version of the command
 *   ranker that we use to rank the intepretations of a disambiguation
 *   response.  
 */
class DisambigRanking: CommandRanking
    /*
     *   Add the ordinal count ranking criterion at the end of the
     *   inherited list of ranking criteria.  If we can't find any
     *   differences on the basis of the other criteria, choose the
     *   interpretation that uses fewer ordinal phrases.  (We prefer an
     *   non-ordinal interpretation, because this will prefer matches to
     *   explicit vocabulary for objects over matches for generic
     *   ordinals.)  
     */
    rankingCriteria = static (inherited() + rankByDisambigOrdinals)
    
    /*
     *   note the an ordinal response is out of range 
     */
    noteOrdinalOutOfRange(ord)
    {
        /* count it as a non-matching entry */
        ++nonMatchCount;
    }

    /* 
     *   note a list ordinal (i.e., "the first one" to refer to the first
     *   item in the ambiguous list) - we take list ordinals as less
     *   desirable than treating ordinal words as adjectives or nouns
     */
    noteDisambigOrdinal()
    {
        /* count it as an ordinal entry */
        ++disambigOrdinalCount;
    }

    /* number of list ordinals in the match */
    disambigOrdinalCount = 0
;

/* ------------------------------------------------------------------------ */
/*
 *   Disambiguation Resolver.  This is a special resolver that we use for
 *   resolving disambiguation responses.  
 */
class DisambigResolver: ProxyResolver
    construct(matchText, ordinalMatchList, matchList, fullMatchList, resolver)
    {
        /* inherit the base class constructor */
        inherited(resolver);
        
        /* remember the original match text and lists */
        self.matchText = matchText;
        self.ordinalMatchList = ordinalMatchList;
        self.matchList = matchList;
        self.fullMatchList = fullMatchList;
    }

    /*
     *   Match an object's name - we'll use the disambiguation name
     *   resolver.  
     */
    matchName(obj, origTokens, adjustedTokens)
    {
        return obj.matchNameDisambig(origTokens, adjustedTokens);
    }

    /*
     *   Resolve qualifiers in the enclosing main scope, since qualifier
     *   phrases in responses are not part of the narrowed list being
     *   disambiguated.  
     */
    getQualifierResolver() { return origResolver; }

    /* 
     *   determine if an object is in scope - it's in scope if it's in the
     *   original full match list 
     */
    objInScope(obj)
    {
        return fullMatchList.indexWhich({x: x.obj_ == obj}) != nil;
    }

    /* for 'all', use the full current full match list */
    getAll() { return fullMatchList; }

    /* filter an ambiguous noun list */
    filterAmbiguousNounPhrase(lst, requiredNum)
    {
        /* 
         *   we're doing disambiguation, so we're only narrowing the
         *   original match list, which we've already filtered as well as
         *   we can - just return the list unchanged 
         */
        return lst;
    }

    /* filter a plural noun list */
    filterPluralPhrase(lst)
    {
        /* 
         *   we're doing disambiguation, so we're only narrowing the
         *   original match list, which we've already filtered as well as
         *   we can - just return the list unchanged 
         */
        return lst;
    }

    /*
     *   Select the match for an indefinite noun phrase.  In interactive
     *   disambiguation, an indefinite noun phrase simply narrows the
     *   list, rather than selecting any match, so treat this as still
     *   ambiguous.  
     */
    selectIndefinite(results, lst, requiredNumber)
    {
        /* note the ambiguous list in the results */
        return results.ambiguousNounPhrase(nil, '', lst, lst, lst,
                                           requiredNumber, self);
    }

    /* the text of the phrase we're disambiguating */
    matchText = ''

    /* 
     *   The "ordinal" match list: this includes the exact list offered as
     *   interactive choices in the same order as they were shown in the
     *   prompt.  This list can be used to correlate ordinal responses to
     *   the prompt list, since it contains exactly the items listed in
     *   the prompt.  Note that this list will only contain one of each
     *   indistinguishable object.  
     */
    ordinalMatchList = []

    /* 
     *   the original match list we are disambiguating, which includes all
     *   of the objects offered as interactive choices, and might include
     *   indistinguishable equivalents of offered items 
     */
    matchList = []

    /* 
     *   the full original match list, which might include items in scope
     *   beyond those offered as interactive choices 
     */
    fullMatchList = []
;

/* ------------------------------------------------------------------------ */
/*
 *   General class for disambiguation exceptions 
 */
class DisambigException: Exception
;

/*
 *   Still Ambiguous Exception - this is thrown when the user answers a
 *   disambiguation question with insufficient specificity, so that we
 *   still have an ambiguous list. 
 */
class StillAmbiguousException: DisambigException
    construct(matchList, origText)
    {
        /* remember the new match list and text */
        matchList_ = matchList;
        origText_ = origText;
    }

    /* the narrowed, but still ambiguous, match list */
    matchList_ = []

    /* the text of the new phrasing */
    origText_ = ''
;

/*
 *   Unmatched disambiguation - we throw this when the user answers a
 *   disambiguation question with a syntactically valid response that
 *   doesn't refer to any of the objects in the list of choices offered. 
 */
class UnmatchedDisambigException: DisambigException
    construct(resp)
    {
        /* remember the response text */
        resp_ = resp;
    }

    /* the response text */
    resp_ = resp;
;


/*
 *   Disambiguation Ordinal Out Of Range - this is thrown when the user
 *   answers a disambiguation question with an ordinal, but the ordinal is
 *   outside the bounds of the offered list (for example, we ask "which
 *   book do you mean, the red book, or the blue book?", and the user
 *   answers "the fourth one"). 
 */
class DisambigOrdinalOutOfRangeException: DisambigException
    construct(ord)
    {
        /* remember the ordinal word */
        ord_ = ord;
    }

    /* a string giving the ordinal word entered by the user */
    ord_ = ''
;

/* ------------------------------------------------------------------------ */
/*
 *   A disambiguation results gatherer object.  We use this to manage the
 *   results of resolution of a disambiguation response.  
 */
class DisambigResults: BasicResolveResults
    construct(parent)
    {
        /* copy the actor information from the parent resolver */
        setActors(parent.targetActor_, parent.issuingActor_);
    }

    ambiguousNounPhrase(keeper, txt, matchList, fullMatchList, scopeList,
                        requiredNum, resolver)
    {
        /*
         *   Our disambiguation response itself requires further
         *   disambiguation.  Do not handle it recursively, since doing so
         *   could allow the user to blow the stack simply by answering
         *   with the same response over and over.  Instead, throw a
         *   "still ambiguous" exception - the original disambiguation
         *   loop will note the situation and iterate on the resolution
         *   list, ensuring that we can run forever without blowing the
         *   stack, if that's the game the user wants to play.  
         */
        throw new StillAmbiguousException(matchList, txt);
    }

    /*
     *   note the an ordinal response is out of range 
     */
    noteOrdinalOutOfRange(ord)
    {
        /* this is an error */
        throw new DisambigOrdinalOutOfRangeException(ord);
    }

    /*
     *   show a message on not matching an object - for a disambiguation
     *   response, failing to match means that the combination of the
     *   disambiguation response plus the original text doesn't name any
     *   objects, not that the object in the response itself isn't present 
     */
    noMatch(txt)
    {
        /* throw an error indicating the problem */
        throw new UnmatchedDisambigException(txt);
    }

    noVocabMatch(txt)
    {
        /* throw an error indicating the problem */
        throw new UnmatchedDisambigException(txt);
    }
;


