/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _MULTILINEDIT_H
#define _MULTILINEDIT_H

#include <vector>

#include <wftk/font.h>
#include <wftk/lineedit.h>

namespace wftk {

/// multi-line text box for user input
class MultiLineEdit : public LineEdit
{
 public:
  /** Constructor.
   *
   * \param text Initial text for this widget.  Defaults to empty string.
   * \param font Initial font for this widget.  Defaults to wftk system font.
   * \param readonly boolean flag to lock / unlock direct text input
   * \param fill boolean flag, if true then this widget will expand to fill
   *        all available space
   */
  explicit MultiLineEdit(const std::string& text = "", const Font &font=Font(),
	bool readonly = true, bool fill = false);
  ///
  virtual ~MultiLineEdit();

  /// Control word-wrap behaviour
  void setWrapping(bool flag);
  /// Set scrollback size
  void setMaxLinesToStore(unsigned n) 
    {maxLinesToStore_ = n; textUpdate();}

  /// append text to this widget
  virtual void addText(const std::string&);  
  /// start a new line on the text output area
  void newline() {addText("\n");}
  /// return the number of visible lines
  unsigned visibleLines() const;
  /// keep the current markup font, but change its color
  void setMarkupColor(unsigned index, const Color& fontCol);
  ///
  void setMarkupFont(unsigned index, const Font& font);
  /// Set background image from a Resource
  void setImage(unsigned index, Surface::Resource* surface);
  /// Set background image from a Surface
  void setImage(unsigned index, const Surface& surface);
  /// Set background image from a named resource
  void setImage(unsigned index, const std::string&);

  /** Markup tags for text formatting / link embedding:
   *  in c-strings you may use the octal encodings:
   *  \177 \176 \174 \173 \172
   */
  enum tag_t { 
    START_TAG=127, ///< Start markup 
	END_TAG=126, ///< End markup
	FONT_TAG=124, ///< Change font inline
	IMAGE_TAG=123, ///< Insert an image (Surface) inline
	LINK_TAG=122 ///< Insert an inline link
  };
  
  /// signal emitted when a link markup is hovered
  SigC::Signal1<void, unsigned> linkActivated;

  /// mouse-click handler function
  virtual bool buttonEvent(Mouse::Button, bool, const Point&);

 protected:
  ///
  virtual void setPackingInfo();
  ///
  virtual void handleResize(Uint16 w, Uint16 h);
  ///
  void breakText(std::string&, std::vector<std::string>&, unsigned);
  ///
  void renderTextLines(const std::vector<std::string>&);
  ///
  unsigned maxLinesToStore_;
 private:
  ///
  void preprocessLines(std::vector<std::string>& lines);
  /// returns new cursor position
  virtual Point drawText(Surface&, const Point&, const Region&);
  ///
  virtual void updateText();
  ///
  unsigned maxLines();
  ///
  bool wordWrap_;
  ///
  bool readOnly_;
  /// whether to try to fill parent() vertically
  bool fill_;
  ///
  std::vector<Font> fonts_;
  ///
  std::vector<Surface::Resource*> images_;
  ///
  typedef struct 
  {
    unsigned link;      // 0 means no link
    unsigned image;     // 0 means no images
    unsigned font;      // 0 means the default font
    std::string text;
  } TextChunk;
  ///
  typedef std::vector<TextChunk> TextLine;
  ///
  std::vector<TextLine> textLines_;
  ///
  unsigned displayLines_;
  ///
  typedef struct
  {
    unsigned id;
    Rect rect;
  } LinkArea;
  ///
  std::vector<LinkArea> linkAreas_;
};

} // namespace wftk

#endif // _MULTILINEDIT_H
