#ifndef WFTK_MAPINIT_H
#define WFTK_MAPINIT_H

// a trick to init a map from constant data
/**
 * It's easy to init a container like std::vector from constant data.
 * For a std::vector<A>, the sequence would be something like:
 *
 * const A init_data[] = {...};
 * const unsigned num_init_data = sizeof(init_data)/sizeof(init_data[0]);
 * const std::vector<A> vec(init_data, init_data + num_init_data);
 *
 * where we use the fact that a pointer is a valid input iterator.
 * It's somewhat harder with a std::map, since std::pair can only
 * initialize from a normal constructor. You have to do something like:
 *
 * const MapType::value_type init_data[] = {MapType::value_type(key1, data1), ...};
 *
 * writing out each of the constructors explicitly. The following class
 * provides a trick to initialize the std::pair values in the map.
 * It should work for any container of std::pair<>.
 * The initialization sequence then looks something like:
 *
 * const PairInit<MapType::value_type> init_data[] = {{key1, data1}, {key2, data2}, ...};
 * const unsigned num_init_data = sizeof(init_data)/sizeof(init_data[0]);
 * const MapType map(init_data->itr(), init_data->itr(num_init_data));
 *
 * We use a custom iterator which casts to the type passed in the template
 * when * dereferenced, so this works out nicely.
 **/

namespace wftk {

template<class P>
struct PairInit
{
  // We can't implement _any_ constructors if we want
  // C-style struct initialization to work. Fortunately,
  // the default constructors (and the default operator=())
  // do what we want them to.

  typename P::first_type first;
  typename P::second_type second;

  operator P() const {return P(first, second);}

  class const_iterator
  {
   public:
    const_iterator(const PairInit* val) : _val(val) {}

    // default copy constructor, operator=() are fine

    P operator*() const {return *_val;}
    // I->first, I->second will work because of this
    const PairInit* operator->() const {return _val;}

    const_iterator& operator++() {++_val; return *this;}
    const_iterator operator++(int) {const_iterator tmp(*this); operator++(); return tmp;}

    bool operator==(const const_iterator& I) const {return _val == I._val;}
    bool operator!=(const const_iterator& I) const {return _val != I._val;}

   private:
    const PairInit* _val;
  };

  const_iterator itr(unsigned offset = 0) const {return this + offset;}
};

} // namespace

#endif // WFTK_MAPINIT_H
