
/*
 * utils.C -- written for Juice
 *  Copyright (C) 1999, 2000, 2001 Abraham vd Merwe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <assert.h>
#include <string.h>

#ifndef CUSTOM_CWD
#include <unistd.h>
#endif

#include "typedefs.h"

/*
 * Canonicalize path, and return a new path. The new path is
 * an absolute path (in it's simplest form). Returns 0 if
 * successful, -1 if an error occurred. check errno to see
 * what error occurred.
 *
 * ERRORS:
 *	  ERANGE     path is to small, increase size (should be PATH_MAX anyway)
 *    ENOMEM     out of memory
 *
 * WORKINGS:
 *    Multiple /'s are collapsed to a single /'.
 *    .'s is replaced with the current working directory.
 *    ..'s are handled by removing portions of the path.
 *    Trailing /'s are removed.
 */
#ifndef CUSTOM_CWD
int canonicalize_pathname (char *path,size_t size)
#else
int canonicalize_pathname (char *path,size_t size,const char *cwd)
#endif
{
   char *buf,*cur,*next;
   int level = 0;
   /* no way path can be smaller than '/' */
   if (size < 2)
	 {
		errno = ERANGE;
		return -1;
	 }
   /* if it's empty, change to CWD */
   if (strlen (path) == 0)
	 {
#ifndef CUSTOM_CWD
		if (getcwd (path,size) == NULL) return -1;
#else
		if (strlen (cwd) >= size) return -1;
		strcpy (path,cwd);
#endif
		return 0;
	 }
   start_from_scratch:
   if ((buf = (char *) malloc ((strlen (path) + 1) * sizeof (char))) == NULL)
	 {
		errno = ENOMEM;
		return -1;
	 }
   cur = path;
   *buf = '\0';
   do
	 {
		if ((next = strchr (cur,'/')) != NULL) *next = '\0', next++;
//fprintf (stderr,"%s: %d: level = %d, cur = \"%s\", next = \"%s\", buf = \"%s\"\n",__FILE__,__LINE__,level,cur,next,buf);
		/* avoid duplicate '/'s */
		if (!strlen (cur) && level)
		  {
			 /* ignore */
		  }
		/* '.' */
		else if (!strcmp (cur,"."))
		  {
			 /* if it's the first level, make it CWD and start from scratch, otherwise ignore */
			 if (!level)
			   {
				  if (next != NULL)
					{
					   strcpy (buf,next);
#ifndef CUSTOM_CWD
					   if (getcwd (path,size) == NULL)
						 {
							free (buf);
							return -1;
						 }
#else
					   if (strlen (cwd) >= size)
						 {
							errno = ERANGE;
							free (buf);
							return -1;
						 }
					   strcpy (path,cwd);
#endif
					   if (strlen (path) + strlen (buf) + (!strcmp (path,"/") ? 0 : 1) >= size)
						 {
							errno = ERANGE;
							free (buf);
							return -1;
						 }
					   if (strcmp (path,"/")) strcat (path,"/");
					   strcat (path,buf);
					   free (buf);
					   goto start_from_scratch;
					}
#ifndef CUSTOM_CWD
					   if (getcwd (buf,size) == NULL)
						 {
							free (buf);
							return -1;
						 }
#else
					   if (strlen (cwd) >= size)
						 {
							errno = ERANGE;
							free (buf);
							return -1;
						 }
					   strcpy (buf,cwd);
#endif
			   }
		  }
		/* '/' */
		else if (!strcmp (cur,""))
		  {
			 /* if it's the first level, copy it, otherwise ignore */
			 if (!level)
			   {
				  strcpy (buf,"/");
				  level++;
			   }
		  }
		/* '..' */
		else if (!strcmp (cur,".."))
		  {
			 /* first level, make it CWD/.. and start from scratch */
			 if (!level)
			   {
				  free (buf);
#ifndef CUSTOM_CWD
				  if (getcwd (path,size) == NULL) return -1;
#else
				  if (strlen (cwd) >= size) return -1;
				  strcpy (path,cwd);
#endif
				  if (strlen (path) + 3 >= size)
					{
					   errno = ERANGE;
					   return -1;
					}
				  strcat (path,"/..");
				  goto start_from_scratch;
			   }
			 /* if it's not the second level step one level back */
			 else if (level != 1)
			   {
				  if ((cur = strrchr (buf,'/')) == NULL) strcpy (buf,*path == '/' ? "/" : ""); else cur[cur == buf ? 1 : 0] = '\0';
				  level--;
			   }
		  }
		/* normal directory name */
		else
		  {
			 if (level && level != 1)
			   {
				  if (strlen (buf) + 1 >= size)
					{
					   free (buf);
					   errno = ERANGE;
					   return -1;
					}
				  strcat (buf,"/");
				  level++;
			   }
			 else if (!strlen (buf))
			   {
				  if (*path == '/') strcpy (buf,"/");
				  level++;
			   }
			 if (strlen (buf) + strlen (cur) >= size)
			   {
				  free (buf);
				  errno = ERANGE;
				  return -1;
			   }
			 strcat (buf,cur);
			 level++;
		  }
		cur = next;
	 }
   while (next != NULL);
   strcpy (path,buf);
   free (buf);
   return 0;
}

