/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <errno.h>
#include "gnome-cmd-includes.h"
#include "gnome-cmd-file-funcs.h"
#include "gnome-cmd-dir-funcs.h"
#include "utils.h"
#include "owner.h"
#include "imageloader.h"
#include "gnome-cmd-data.h"
#include "gnome-cmd-file-props-dialog.h"
#include "gnome-cmd-main-win-types.h"
#include "gnome-cmd-xfer.h"
#include "dir_pool.h"

#define MAX_TYPE_LENGTH 2
#define MAX_NAME_LENGTH 128
#define MAX_OWNER_LENGTH 128
#define MAX_GROUP_LENGTH 128
#define MAX_PERM_LENGTH 10
#define MAX_DATE_LENGTH 64
#define MAX_SIZE_LENGTH 32

int file_counter = 0;

static GtkObjectClass *parent_class = NULL;


/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	GnomeCmdFile *file = GNOME_CMD_FILE (object);

		DEBUG ('f', "file destroying 0x%x %s\n", (guint)file, file->info->name);
		gnome_vfs_file_info_unref (file->info);
		gnome_vfs_uri_unref (file->uri);
		g_free (file->uri_str);
		g_free (file->path);
		file_counter--;
	
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
class_init (GnomeCmdFileClass *class)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (class);
	parent_class = gtk_type_class (gtk_object_get_type ());

	object_class->destroy = destroy;
}

static void
init (GnomeCmdFile *file)
{
	file->ref_cnt = 0;
	file->info = NULL;
	file->dir = NULL;
	file->uri = NULL;
	file->uri_str = NULL;
	file_counter++;
}



/***********************************
 * Public functions
 ***********************************/

GtkType
gnome_cmd_file_get_type         (void)
{
	static GtkType type = 0;

	if (type == 0)
	{
		GtkTypeInfo info =
		{
			"GnomeCmdFile",
			sizeof (GnomeCmdFile),
			sizeof (GnomeCmdFileClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (gtk_object_get_type (), &info);
	}
	return type;
}


GnomeCmdFile*
gnome_cmd_file_new (void)
{
	GnomeCmdFile *file;

	file = gtk_type_new (gnome_cmd_file_get_type ());

	return file;
}


GnomeCmdFile *gnome_cmd_file_new_with_values (GnomeVFSFileInfo *info, GnomeCmdDir *dir)
{
	GnomeCmdFile *file = gnome_cmd_file_new ();

	g_return_val_if_fail (file != NULL, NULL);
	
	file->info = info;
	file->dir = dir;

	file->uri = gnome_cmd_dir_get_file_uri (dir, info->name);
	file->uri_str = gnome_vfs_uri_to_string (file->uri, 0);
	file->path = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (file->uri), 0);
	
	gnome_vfs_file_info_ref (file->info);

	return file;
}


void gnome_cmd_file_ref (GnomeCmdFile *file)
{
	g_return_if_fail (file != NULL);

	file->ref_cnt++;
	
	DEBUG ('f', "file refing: 0x%x %s to %d\n", (guint)file, file->info->name, file->ref_cnt);
}


void gnome_cmd_file_unref (GnomeCmdFile *file)
{
	g_return_if_fail (file != NULL);
	
	file->ref_cnt--;
	
	DEBUG ('f', "file un-refing: 0x%x %s to %d\n", (guint)file, file->info->name, file->ref_cnt);
	if (file->ref_cnt < 1)
		gtk_object_unref (GTK_OBJECT (file));
}


GnomeVFSResult gnome_cmd_file_chmod (GnomeCmdFile *file, GnomeVFSFilePermissions perm)
{
	GnomeVFSResult ret;

	g_return_val_if_fail (file != NULL, 0);
	g_return_val_if_fail (file->info != NULL, 0);

	file->info->permissions = perm;
	ret = gnome_vfs_set_file_info_uri (
		gnome_cmd_file_get_uri (file),
		file->info,
		GNOME_VFS_SET_FILE_INFO_PERMISSIONS);

	return ret;
}


GnomeVFSResult gnome_cmd_file_chown (GnomeCmdFile *file, uid_t uid, gid_t gid)
{
	GnomeVFSResult ret;

	g_return_val_if_fail (file != NULL, 0);
	g_return_val_if_fail (file->info != NULL, 0);

	if (uid != -1)
		file->info->uid = uid;
	file->info->gid = gid;
	ret = gnome_vfs_set_file_info_uri (
		gnome_cmd_file_get_uri (file),
		file->info,
		GNOME_VFS_SET_FILE_INFO_OWNER);

	return ret;
}


GnomeVFSResult gnome_cmd_file_rename (GnomeCmdFile *finfo, const gchar *new_name)
{
	GnomeVFSFileInfo *new_info;
	GnomeVFSResult result;

	g_return_val_if_fail (finfo, FALSE);
	g_return_val_if_fail (finfo->info, FALSE);

	new_info = gnome_vfs_file_info_dup (finfo->info);
	g_return_val_if_fail (new_info, FALSE);

	g_free (new_info->name);
	new_info->name = g_strdup (new_name);
	
	result = gnome_vfs_set_file_info_uri (
		gnome_cmd_file_get_uri (finfo),
		new_info,
		GNOME_VFS_SET_FILE_INFO_NAME);

	if (result == GNOME_VFS_OK) {
		if (!gnome_cmd_dir_uses_fam (finfo->dir)) {
			gnome_cmd_file_update_info (finfo, new_info);
			gnome_cmd_dir_file_renamed (finfo->dir, finfo);
		}
	}

	return result;
}


const gchar *gnome_cmd_file_get_name (GnomeCmdFile *file)
{
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	return file->info->name;
}


gchar *gnome_cmd_file_get_quoted_name (GnomeCmdFile *file)
{
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	return quote_if_needed (file->info->name);
}


const gchar *gnome_cmd_file_get_extension (GnomeCmdFile *file)
{
	gint i, len;
	
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);

	if (file->info->type == GNOME_VFS_FILE_TYPE_DIRECTORY)
		return NULL;

	len = strlen (file->info->name);
	for ( i=len ; i>0 ; i-- ) {
		if (file->info->name[i] == '.')
			return &file->info->name[i+1];
	}

	return NULL;
}


const gchar *gnome_cmd_file_get_owner (GnomeCmdFile *file)
{
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	if (GNOME_VFS_FILE_INFO_LOCAL (file->info))
	{
		user_t *owner = OWNER_get_user_by_uid (file->info->uid);
		return owner->name;
	}
	else
	{
		static gchar owner_str[MAX_OWNER_LENGTH];
		g_snprintf (owner_str, MAX_OWNER_LENGTH, "%d", file->info->uid);
		return owner_str;
	}
}


const gchar *gnome_cmd_file_get_group (GnomeCmdFile *file)
{
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	if (GNOME_VFS_FILE_INFO_LOCAL (file->info))
	{
		group_t *group = OWNER_get_group_by_gid (file->info->gid);
		return group->name;
	}
	else
	{
		static gchar group_str[MAX_GROUP_LENGTH];
		g_snprintf (group_str, MAX_GROUP_LENGTH, "%d", file->info->gid);
		return group_str;
	}
}


static const gchar *date2string (time_t date,
								 gboolean overide_disp_setting)
{
	return time2string (date, overide_disp_setting?"%c":gnome_cmd_data_get_date_format ());
}




const gchar *gnome_cmd_file_get_adate (GnomeCmdFile *file,
									   gboolean overide_disp_setting)
{
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	return date2string (file->info->atime, overide_disp_setting);
}


const gchar *gnome_cmd_file_get_mdate (GnomeCmdFile *file,
									   gboolean overide_disp_setting)
{
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	return date2string (file->info->mtime, overide_disp_setting);
}


const gchar *gnome_cmd_file_get_cdate (GnomeCmdFile *file,
									   gboolean overide_disp_setting)
{
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	return date2string (file->info->ctime, overide_disp_setting);
}


const gchar *gnome_cmd_file_get_size (GnomeCmdFile *file)
{
	static gchar dir_indicator[] = "<DIR> ";
	
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	if (file->info->type == GNOME_VFS_FILE_TYPE_DIRECTORY)
		return dir_indicator;

	return size2string (file->info->size, gnome_cmd_data_get_size_disp_mode ());
}


const gchar *gnome_cmd_file_get_tree_size (GnomeCmdFile *file)
{
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);

	if (file->info->type != GNOME_VFS_FILE_TYPE_DIRECTORY)
		return gnome_cmd_file_get_size (file);

	if (strcmp (file->info->name, "..") == 0)
		return gnome_cmd_file_get_size (file);

	return size2string (calc_tree_size (gnome_cmd_file_get_uri (file)),
						gnome_cmd_data_get_size_disp_mode ());
}


const gchar *gnome_cmd_file_get_perm (GnomeCmdFile *file)
{
	static gchar perm_str[MAX_PERM_LENGTH];

	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	perm2string (file->info->permissions, perm_str, MAX_PERM_LENGTH);
	return perm_str;
}


const gchar *gnome_cmd_file_get_type_string (GnomeCmdFile *finfo)
{
	static gchar type_str[MAX_TYPE_LENGTH];
	
	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);
	
	type2string (finfo->info->type, type_str, MAX_TYPE_LENGTH);
	return type_str;
}


const gchar *gnome_cmd_file_get_type_desc (GnomeCmdFile *finfo)
{
	gchar *type_strings[] = {
		_("Unknown file type"),
		_("Regular file"),
		_("Directory"),
		_("FIFO"),
		_("UNIX Socket"),
		_("Character device"),
		_("Block device"),
		_("Symbolic link")
	};

	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);

	if (!finfo->info->symlink_name)	
		return type_strings[finfo->info->type];
	return type_strings[GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK];
}


GdkPixmap *gnome_cmd_file_get_type_pixmap (GnomeCmdFile *finfo)
{
	GdkPixmap *pm;
	GdkBitmap *mask;
	
	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);

	IMAGE_get_pixmap_and_mask (finfo->info->type,
							   finfo->info->mime_type,
							   finfo->info->symlink_name != NULL,
							   &pm, &mask);
	return pm;
}


GdkBitmap *gnome_cmd_file_get_type_mask (GnomeCmdFile *finfo)
{
	GdkPixmap *pm;
	GdkBitmap *mask;
	
	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);

	IMAGE_get_pixmap_and_mask (finfo->info->type,
							   finfo->info->mime_type,
							   finfo->info->symlink_name != NULL,
							   &pm, &mask);
	return mask;
}


const gchar *gnome_cmd_file_get_path (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);

	return finfo->path;
}


gchar *gnome_cmd_file_get_quoted_path (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);
	
	return quote_if_needed (finfo->path);
}


GnomeVFSURI *gnome_cmd_file_get_uri (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);
	
	return finfo->uri;	
}


const gchar *gnome_cmd_file_get_uri_str (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);

	return finfo->uri_str;
}


const gchar *gnome_cmd_file_get_mime_type (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);

	return finfo->info->mime_type;
}


const gchar *gnome_cmd_file_get_mime_type_desc (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);
	
	return gnome_vfs_mime_get_description (finfo->info->mime_type);
}


const gchar *gnome_cmd_file_get_dir_path (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);

	return gnome_cmd_dir_get_path (finfo->dir);
}


void gnome_cmd_file_show_properties (GnomeCmdFile *finfo)
{
	GtkWidget *dialog = gnome_cmd_file_props_dialog_create (finfo);
	if (!dialog) return;
	gtk_widget_ref (dialog);
	gtk_object_set_data_full (
		GTK_OBJECT (main_win), "gnome_cmd_file_props_dialog", dialog,
		(GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog);
}


static void
do_view_file (const gchar *path)
{
	const gchar *viewer;
	gchar *command;

	viewer = gnome_cmd_data_get_viewer ();
	
	command = g_strdup_printf (viewer, path);
	run_command (command, FALSE);
	
	g_free (command);
}


static void
on_file_downloaded_for_view (gchar *path)
{
	do_view_file (path);
	
	g_free (path);
}


void gnome_cmd_file_view (GnomeCmdFile *finfo)
{
	gchar *path;

	gchar *dest_uri_path;
	GnomeVFSURI *src_uri, *dest_uri;

	g_return_if_fail (finfo != NULL);

	/* If the file is local there is no need to download it */
	if (gnome_vfs_uri_is_local (gnome_cmd_file_get_uri (finfo))) {
		do_view_file (gnome_cmd_file_get_path  (finfo));
		return;
	}

	/* The file is remove lets download it to a temporary file first */
	path = get_temp_download_filepath (gnome_cmd_file_get_name (finfo));
	if (!path) return;

	src_uri = gnome_vfs_uri_dup (gnome_cmd_file_get_uri (finfo));
	dest_uri_path = g_strdup_printf ("file://%s", path);
	dest_uri = gnome_vfs_uri_new (dest_uri_path);
	g_free (dest_uri_path);

	g_printerr ("Copying to: %s\n", path);
	
	gnome_cmd_xfer_tmp_download (src_uri,
								 dest_uri,
								 GNOME_VFS_XFER_FOLLOW_LINKS,
								 GNOME_VFS_XFER_OVERWRITE_MODE_REPLACE,
								 on_file_downloaded_for_view,
								 path);
}


void gnome_cmd_file_edit (GnomeCmdFile *finfo)
{
	gchar *command;
	
	g_return_if_fail (finfo != NULL);
	
	command = g_strdup_printf (gnome_cmd_data_get_editor (), gnome_cmd_file_get_path (finfo));
	run_command (command, FALSE);
	g_free (command);
}


void gnome_cmd_file_show_cap_cut (GnomeCmdFile *finfo)
{
}


void gnome_cmd_file_show_cap_copy (GnomeCmdFile *finfo)
{
}


void gnome_cmd_file_show_cap_paste (GnomeCmdFile *finfo)
{
}


void gnome_cmd_file_update_info (GnomeCmdFile *finfo, GnomeVFSFileInfo *info)
{
	g_return_if_fail (finfo != NULL);
	g_return_if_fail (info != NULL);
	
	gnome_vfs_file_info_unref (finfo->info);
	gnome_vfs_file_info_ref (info);
	finfo->info = info;

	if (finfo->uri)
		gnome_vfs_uri_unref (finfo->uri);
	finfo->uri = gnome_cmd_dir_get_file_uri (finfo->dir, info->name);

	if (finfo->uri_str)
		g_free (finfo->uri_str);	
	finfo->uri_str = gnome_vfs_uri_to_string (finfo->uri, 0);

	if (finfo->path)
		g_free (finfo->path);
	finfo->path = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (finfo->uri), 0);
}


/******************************************************************************
*
*   Function: gnome_cmd_file_list_copy
*
*   Purpose: Refs all files in the passed list and return a copy of that list
*
*   Params: 
*
*   Returns: A copy of the passed list with all files ref'ed
*
*   Statuses: 
*
******************************************************************************/

GList *gnome_cmd_file_list_copy (GList *files)
{
	g_return_val_if_fail (files != NULL, NULL);

	gnome_cmd_file_list_ref (files);
	return g_list_copy (files);
}


/******************************************************************************
*
*   Function: gnome_cmd_file_list_free
*
*   Purpose: Unrefs all files in the passed list and then frees the list
*
*   Params: 
*
*   Returns: 
*
*   Statuses: 
*
******************************************************************************/

void gnome_cmd_file_list_free (GList *files)
{
	if (!files) return;

	gnome_cmd_file_list_unref (files);
	g_list_free (files);
}


/******************************************************************************
*
*   Function: gnome_cmd_file_list_ref
*
*   Purpose: Refs all files in the passed list
*
*   Params: 
*
*   Returns: 
*
*   Statuses: 
*
******************************************************************************/

void gnome_cmd_file_list_ref (GList *files)
{
	g_return_if_fail (files != NULL);

	g_list_foreach (files, (GFunc)gnome_cmd_file_ref, NULL);
}


/******************************************************************************
*
*   Function: gnome_cmd_file_list_unref
*
*   Purpose: Unrefs all files in the passed list
*
*   Params: 
*
*   Returns: 
*
*   Statuses: 
*
******************************************************************************/

void gnome_cmd_file_list_unref (GList *files)
{
	g_return_if_fail (files != NULL);

	g_list_foreach (files, (GFunc)gnome_cmd_file_unref, NULL);
}



