/*  GFC-Core: GTK+ Foundation Classes (Core Library)
 *  Copyright (C) 2004 The GFC Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/// @file gfc/glib/objectsignals.hh
/// @brief G::Object virtual signal handlers.
///
/// Provides G::ObjectSignals, an abstract base class that implements
/// the overridable virtual signal handlers for G::Object.

#ifndef GFC_G_OBJECT_SIGNALS_HH
#define GFC_G_OBJECT_SIGNALS_HH

#ifndef GFC_G_TYPE_HH
#include <gfc/glib/type.hh>
#endif

namespace GFC {

namespace G {

/// @class ObjectSignals objectsignals.hh gfc/glib/objectsignals.hh
/// @brief Abstract base class that implements the virtual signal handlers for G::Object.
///
/// In GFC a signal object is an abstract implementation class that implements the
/// vitual signal handling mechanism for the corresponding instance object. 
/// Signal objects can only be used as a base class to multiplely inherit from.
/// This way, the overhead associated with multiple virtual function calls and
/// large virtual function tables is minimized by not forcing you to accept
/// this overhead when your not overriding any virtual signal handlers.
///
/// The main reason to override an object's virtual signal handlers would be to
/// customize an object's default response to one or more signals. You should note
/// that in GFC each virtual signal handler is called before the corresponding default
/// GTK+ signal handler. This gives you the powerful choice of either calling or not 
/// calling the default handler. Another reason to override an object's virtual signal
/// handlers would be for convenience. When your deriving a new class from an exisiting 
/// object instead of connecting to object signals in your constructor using the object's
/// proxy signal functions (ligsigc) you can override one or more of the object's virtual
/// signal handlers by multiplely inheriting your new class from the object's signal class. 
/// For example, to create a main window and override the on_delete_event() signal handler
/// you would do something like this:
/// 
/// @code
/// #include <gfc/gtk/window.hh>
/// #include <gfc/gtk/windowsignals.hh>
///
/// using namespace GFC;
///
/// class MyWindow : public Gtk::Window, protected Gtk::WindowSignals
/// {
///     virtual bool on_delete_event(const Gdk::EventAny& event);
///
/// public: 
///     MyWindow();
/// };

/// MyWindow::MyWindow()
/// : Gtk::WindowSignals(*this)
/// {
///     // put your code here
/// }
/// 
/// bool 
/// MyWindow::on_delete_event(const Gdk::EventAny& event)
/// {
///     // put your code here     
/// 
///     return true;
/// }
/// @endcode

class ObjectSignals : public virtual TypeInstance
{
protected:
/// @name Constructors
/// @{

	ObjectSignals(Object *object);
	///< Constructs a new ObjectSignals object.
	///< @param object A G::Object inheriting the G::ObjectSignals implementation.

	virtual ~ObjectSignals() = 0;
	///< Destructor.

/// @}

public:
/// @name Signal Handlers
/// @{

	virtual void on_notify(GParamSpec *pspec);
	///< Called when a property on an object is changed.
	///< @param pspec A GParamSpec object that holds the meta data specifying the new property.

/// @}

	static GQuark quark();
	///< Returns a quark that indentifies an instance object's stored signal
	///< class pointer. The pointer will be null if an object or widget doesn't
	///< multilplely inherit from its signal class. 
};

} // namespace G

} // namespace GFC

#endif // GFC_G_OBJECT_SIGNALS_HH

