/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  Author: Stéphane Démurget  <stephane.demurget@enst-bretagne.fr>
 */

#include "config.h"

#include <gnome.h>

#include <glib.h>
#include <glib/gi18n.h>
#include <gtk/gtkimage.h>

#include "ef-popup.h"
#include "ef-backend.h"
#include "ef-msg-queue.h"
#include "ef-utils.h"
#include "emifreq.h"

enum {
	PERFORMANCE = -1,
	POWERSAVE = -2
};

static void
menu_item_activate_cb (GtkMenuItem *menu_item, gpointer data)
{
	EmiFreqApplet *applet = (EmiFreqApplet *) data;
	int frequency = *((int *) g_object_get_data (G_OBJECT (menu_item), "frequency"));
	EFMessage msg;

	switch (frequency) {
		case PERFORMANCE: msg.type = GOV_PERFORMANCE; break;
		case POWERSAVE:   msg.type = GOV_POWERSAVE;   break;
		default:	  msg.type = GOV_USERSPACE;   break;
	}

	msg.frequency = frequency;

	if (ef_msg_queue_send (applet->msg_queue, &msg))
		force_refresh ((EmiFreqApplet *) data, TRUE);
	else
		display_error (_("Can't change the CPU speed."),
			       _("An error occured while the applet tried to send a command to the daemon in charge of changing the CPU speed. You need to restart the daemon"));
}

static GtkWidget *
construct_menu_item (EmiFreqApplet *applet, int num, float u, char *label)
{
	GdkPixbuf *pixbuf = get_pixbuf_from_usage (applet, u);
	GtkWidget *image = gtk_image_new_from_pixbuf (pixbuf);
	GtkWidget *menu_item;
	int *frequency;

	frequency = g_new (int, 1);
	*frequency = num;

	g_object_unref (pixbuf);

	menu_item = gtk_image_menu_item_new_with_label (label);

	 // Note: even looking at gobject's code I am not sure if g_data_set would do the same ...
	g_object_set_data_full (G_OBJECT (menu_item), "frequency", frequency, g_free);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menu_item), image);

	g_signal_connect (menu_item, "activate", G_CALLBACK (menu_item_activate_cb), applet); 

	return menu_item;
}

static void
set_menu_position (GtkMenu *menu, gint *x, gint *y, gboolean *push_in, gpointer user_data)
{
	GtkWidget *applet;
	GdkScreen *screen;
	GtkRequisition requisition;
	PanelAppletOrient orient;
	gint menu_x = 0;
	gint menu_y = 0;

	g_return_if_fail (user_data != NULL);

	applet = GTK_WIDGET (user_data);
	screen = gtk_widget_get_screen (applet);
	gtk_widget_size_request (GTK_WIDGET (menu), &requisition);
	gdk_window_get_origin (applet->window, &menu_x, &menu_y);

	menu_x += applet->allocation.x;
	menu_y += applet->allocation.y;

	orient = panel_applet_get_orient (PANEL_APPLET (applet));

	if (orient == PANEL_APPLET_ORIENT_UP || orient == PANEL_APPLET_ORIENT_DOWN) {
		if (menu_y > gdk_screen_get_height (screen) / 2)
			menu_y -= requisition.height;
		else
			menu_y += applet->allocation.height;
	} else {
		if (menu_x > gdk_screen_get_width (screen) / 2)
		       	menu_x -= requisition.width;
		else
			menu_x += applet->allocation.width;
	}

	*x = menu_x;
	*y = menu_y;
	*push_in = TRUE;
}

void
ef_popup_show (EmiFreqApplet *applet, guint32 activate_time)
{
	g_return_if_fail (applet != NULL);

	if (!applet->popup || applet->popup_dirty) {
		EFBackendGovernors avail_governors = ef_backend_get_avail_governors ();

		applet->popup = gtk_menu_new ();

		if (((avail_governors & GOV_PERFORMANCE) == GOV_PERFORMANCE))
			gtk_menu_shell_append (GTK_MENU_SHELL (applet->popup),
					       construct_menu_item (applet, PERFORMANCE, 100.0, _("Performance")));

		if ((avail_governors & GOV_USERSPACE) == GOV_USERSPACE) {
			GPtrArray *freqs = ef_backend_get_avail_frequencies ();
			int i;

			for (i = freqs->len - 2; i > 0; i--) {
				gchar *label = g_strdup_printf ("%d Mhz", (*((int *) freqs->pdata[i])) / 1000);

				gtk_menu_shell_append (GTK_MENU_SHELL (applet->popup),
						       construct_menu_item (applet, *((int *) freqs->pdata[i]), 100.0 * i / freqs->len, label));

				g_free (label);
			}
		}

		if ((avail_governors & GOV_POWERSAVE) == GOV_POWERSAVE)
			gtk_menu_shell_append (GTK_MENU_SHELL (applet->popup),
					       construct_menu_item (applet, POWERSAVE, 0.0, _("Powersaving")));

		gtk_widget_show_all (GTK_WIDGET (applet->popup));

		applet->popup_dirty = FALSE;
	}

	gtk_menu_popup (GTK_MENU (applet->popup), NULL, NULL, (GtkMenuPositionFunc) set_menu_position, applet, 1, activate_time);
}
