/*
 * savedump.c - Copies an lkcd raw dump to the file system
 * Created by:	John Daley (John.Daley@hp.com)
 * Minor mods:	Bob Montgomery (bobm@fc.hp.com)
 * Mods:	Troy Heber (troy.heber@hp.com)
 *
 * Copyright 2004 Hewlett-Packard Development Company, L.P.
 * 
 * Adapted from lkcdutils
 * Created by: Matt D. Robinson (yakker@aparity.com)
 * Copyright 2001 Matt D. Robinson (yakker@aparity.com), all rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */ 


#define MAX_LINE_SIZE 132

#include "savedump.h"
#include "parseconf.h"
#include "version.h"

struct confInfo{
    char  *dev;
    char  *dir;
    char  *user;
    char  *pass;
    int   ftp;
    char  *error;
    char  *host;
    char  *path;
    int    port;
};


FILE * do_open(char *);
int parseUrl(char *, char **, char **, char **, int *, char **);

void usage(char *);
void version(char *);
int check_dev( char *);
int reset_dump( char *);
int erase_dump(char *);
int copy_dump(struct confInfo *, int);
static int direxists(const char *);

#if _LIBC
# define struct_stat64 struct stat64
#else
# define struct_stat64 struct stat
# define __xstat64(version, path, buf) stat (path, buf)
#endif


static int
kver(void)
{
	// Check kernel version, assume 2.6 if we can't 
	// figure it out. 
	FILE *fp;
	char buf[512];

	if ( ((fp = fopen("/proc/version", "r")) == NULL)){
		fprintf(stderr, "WARNING: can not read /proc/version\n");
		return 6;
	}

	while (fgets (buf, 512, fp) != NULL){
		if ( strstr (buf, " 2.4.")  != NULL)
			return 4;
		else if ( strstr (buf, " 2.6.")  != NULL)
			return 6;
	}

	return 6;
}

/* Return nonzero if DIR is an existent directory.  */
static int
direxists (const char *dir)
{
    struct_stat64 buf;
    return __xstat64 (_STAT_VER, dir, &buf) == 0 && S_ISDIR (buf.st_mode);
}


// usage message
//
void
usage(char * pname)
{
    fprintf(stderr, "\nUsage:\n");
    fprintf(stderr, "%s [[-r dumpdevice] || [-e dumpdevice]] [-d dumpdevice] [-o dump-path]\n\n", pname);
    fprintf(stderr, "  dumpdevice\n");
    fprintf(stderr, "    The dumpdevice arguement is the dump device that holds the raw LKCD dump.\n");
    fprintf(stderr, "    such as: /dev/vmdump\n\n");
    fprintf(stderr, "  dump-path\n");
    fprintf(stderr, "    The dump-path arguement is the path to store the resulting dump file to.\n");
    fprintf(stderr, "    It may be a path on a filesystem or a FTP URL in the form.\n");
    fprintf(stderr, "    ftp://user:password@host:port/path where user, password,\n"); 
    fprintf(stderr, "    port & path are optional.\n\n");
    fprintf(stderr, "  Optional Arguements:\n\n");
    fprintf(stderr, "  [-r]\n");
    fprintf(stderr, "    Reset the raw LKCD dump from the dumpdevice, option is exclusive of -e\n\n");
    fprintf(stderr, "  [-e]\n");
    fprintf(stderr, "    Erase the raw LKCD dump from the dumpdevice, option is exclusive of -r\n\n");
    fprintf(stderr, "  [-d]\n");
    fprintf(stderr, "    Specify the dumpdevice overriding DUMPDEV= /etc/dumputils.conf file.\n\n");
    fprintf(stderr, "  [-o]\n");
    fprintf(stderr, "    Specify the dump-path overriding DUMPDIR= in the /etc/dumputils.conf file.\n\n");
    fprintf(stderr, "  EXAMPLES:\n\n");
    fprintf(stderr, "       Default: use /etc/dumputils.conf\n");
    fprintf(stderr, "       $ %s \n", pname);
    fprintf(stderr, "       $ %s -r /dev/{dumpdevice}\n", pname);
    fprintf(stderr, "       $ %s -e /dev/{dumpdevice}\n", pname);
    fprintf(stderr, "       $ %s -d /dev/{dumpdevice}\n", pname); 
    fprintf(stderr, "       $ %s -o /var/log/dump\n", pname); 
    fprintf(stderr, "       $ %s -o ftp://user:pass@none.com/pub/dump\n", pname); 
    fprintf(stderr, "       $ %s -d /dev/{dumpdevice} -o ftp://user:pass@none.com/pub/dump\n", pname); 
}

void version(char *pname)
{
#ifdef VERSION
    fprintf(stdout, "%s %g\n", pname, VERSION);
#else
    fprintf(stdout, "%s undefined\n", pname);
#endif 
}


void
percent_done(int sofar, int pages)
{
    unsigned int percent;
    unsigned static int lastpercent;

    percent = (float)sofar / (float)pages * 100.0;
    if (percent == lastpercent)
	return;
    if (percent % 10 == 0) {
	printf("--->%u%%", percent);
	fflush(stdout);
    }

    if(percent >= 100)
	printf("\n\n");

    lastpercent = percent;
    return;
}

// create a bounds file.
//
int
copy_map(struct confInfo *conf, int bounds)
{
    char *smname;
    FILE *dfd=NULL, *fp=NULL;
    char buf[10], line[255], *fuser=NULL, *fpass=NULL;
    int err=0;

    disconnect();

    if(conf->ftp){

	if(conf->path == NULL){
	    if  ((smname=(char *)malloc(10)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(smname, "%d/map.%d", bounds, bounds);
	}else{
	    if  ((smname=(char *)malloc(strlen(conf->path) + 10)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(smname, "%s/%d/map.%d", conf->path, bounds, bounds);
	}

	if  ((fuser=(char *)malloc(strlen(conf->user)+7)) == NULL){
	    perror("FATAL ERROR: Couldn't allocate enough memory");
	    exit(10);
	}

	if  ((fpass=(char *)malloc(strlen(conf->pass)+7)) == NULL){
	    perror("FATAL ERROR: Couldn't allocate enough memory");
	    exit(10);
	}

	sprintf(fuser, "USER %s", conf->user);
	sprintf(fpass, "PASS %s", conf->pass);

	if ( ((err = FTPConnect(conf->host, fuser, fpass)) < 0) ){
	    fprintf(stderr, "Could not connect to server!\n");
	    free(fuser);
	    free(fpass);
	    free(smname);
	    exit(err);
	}

	if ( (dfd = do_open(smname)) == NULL ){
	    fprintf(stderr, "Could not create file!\n");
	    free(fuser);
	    free(fpass);
	    free(smname);
	    disconnect();
	    return 39;
	}

	if((fp = fopen("/boot/System.map", "r")) != NULL){
	    while (fgets(line,sizeof(line),fp) != NULL)
	    {
		fprintf(dfd, "%s", line);
	    }
	    fflush(dfd);
	}else{
	    fprintf(stderr, "Couldn't open /boot/System.map!\n");
	    free(fuser);
	    free(fpass);
	    free(smname);
	    disconnect();
	    return 82;
	}

#ifdef DEBUG
	printf("Wrote System Map: %s\n", smname);
#endif

	fflush(dfd);
	fclose(fp);
	free(fuser);
	free(fpass);
	free(smname);
	disconnect();

	return 0;

    } else {

	if(conf->dir == NULL){
	    if  ((smname=(char *)malloc(20)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(smname, "%d/map.%d", bounds, bounds);

	}else{
	    if  ((smname=(char *)malloc(strlen(conf->dir) + 20)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(smname, "%s/%d/map.%d", conf->dir, bounds, bounds);
	}


#ifdef DEBUG
	printf("System Map file name: %s\n", smname);
#endif
	if ((dfd = fopen(smname, "w")) == NULL) {
	    perror("Error opening destionation System.map file");
	    free(smname);
	    return 81;
	}

	if((fp = fopen("/boot/System.map", "r")) != NULL){
	    while (fgets(line,sizeof(line),fp) != NULL)
	    {
		fprintf(dfd, "%s", line);
	    }
	    fflush(dfd);
	}else{
	    fprintf(stderr, "Couldn't open /boot/System.map\n");
	    free(smname);
	    disconnect();
	    return 82;
	}

#ifdef DEBUG
	printf("Wrote System.map file: %s\n", smname);
#endif

	fflush(dfd);
	fclose(fp);
	fclose(dfd);
	free(smname);
	return 0;
    }

}

// create a bounds file.
//
int
createbounds(struct confInfo *conf, char *num)
{
    char *bfname;
    FILE *dfd;
    char buf[10], *fuser, *fpass;
    int err=0;

    disconnect();

    if(conf->ftp){

	if(conf->path == NULL){
	    if  ((bfname=(char *)malloc(10)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(bfname, "bounds");
	}else{
	    if  ((bfname=(char *)malloc(strlen(conf->path) + 10)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(bfname, "%s/bounds", conf->path);
	}

	if  ((fuser=(char *)malloc(strlen(conf->user)+7)) == NULL){
	    perror("FATAL ERROR: Couldn't allocate enough memory");
	    exit(10);
	}

	if  ((fpass=(char *)malloc(strlen(conf->pass)+7)) == NULL){
	    perror("FATAL ERROR: Couldn't allocate enough memory");
	    exit(10);
	}

	sprintf(fuser, "USER %s", conf->user);
	sprintf(fpass, "PASS %s", conf->pass);

	if ( ((err = FTPConnect(conf->host, fuser, fpass)) != 0) ){
	    fprintf(stderr, "Could not connect to server!\n");
	    free(fuser);
	    free(fpass);
	    free(bfname);
	    exit(err);
	}


	if ( (dfd = do_open(bfname)) == NULL ){
	    fprintf(stderr, "Could not create file!\n");
	    free(fuser);
	    free(fpass);
	    free(bfname);
	    disconnect();
	    return 39;
	}

	if ((fprintf(dfd, "%s\n", num)) < 1){
	    fprintf(stderr, "Could write to bounds file!\n");
	    free(fuser);
	    free(fpass);
	    free(bfname);
	    disconnect();
	    return 39;
	}

#ifdef DEBUG
	printf("Wrote bounds file: %s\n", bfname);
#endif

	fflush(dfd);
	free(fuser);
	free(fpass);
	free(bfname);
	disconnect();

	return 0;

    } else {

	if(conf->dir == NULL){
	    if  ((bfname=(char *)malloc(10)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(bfname, "bounds");
	}else{
	    if  ((bfname=(char *)malloc(strlen(conf->dir) + 10)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(bfname, "%s/bounds", conf->dir);
	}

#ifdef DEBUG
	printf("bounds file name: %s\n", bfname);
#endif
	if ((dfd = fopen(bfname, "w")) == NULL) {
	    perror("Opening bounds file for write.");
	    free(bfname);
	    return 81;
	}

	if ((fprintf(dfd, "%s\n", num)) < 1){
	    fprintf(stderr, "Could write to bounds file!\n");
	    free(bfname);
	    disconnect();
	    return 81;
	}

#ifdef DEBUG
	printf("Wrote bounds file: %s\n", bfname);
#endif

	fflush(dfd);
	fclose(dfd);
	free(bfname);
	return 0;
    }

}

// read bounds, it's part of output file path
//
int
getbounds(struct confInfo *conf)
{
    int boundsf, bounds=-1, ret, err=0;
    char *bfname;
    char buf[10], *fuser, *fpass;

    if(conf->ftp){

	if(conf->path == NULL){
	    if  ((bfname=(char *)malloc(10)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(bfname, "bounds");
	}else{
	    if  ((bfname=(char *)malloc(strlen(conf->path) + 10)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(bfname, "%s/bounds", conf->path);
	}

	if  ((fuser=(char *)malloc(strlen(conf->user)+7)) == NULL){
	    perror("FATAL ERROR: Couldn't allocate enough memory");
	    exit(10);
	}

	if  ((fpass=(char *)malloc(strlen(conf->pass)+7)) == NULL){
	    perror("FATAL ERROR: Couldn't allocate enough memory");
	    exit(10);
	}

	sprintf(fuser, "USER %s", conf->user);
	sprintf(fpass, "PASS %s", conf->pass);

	if ( ((err = FTPConnect(conf->host, fuser, fpass)) != 0) ){
	    fprintf(stderr, "Could not connect to server!\n");
	    exit(err);
	}

	free(fuser);
	free(fpass);

	if((ret = do_readFirstLine(bfname, buf, 9)) < 0){
	    fprintf(stderr, "Could not read: %s\nWill try to create it\n", bfname);
	    free(bfname);
	    return (ret);
	}

	free(bfname);
	disconnect();
	bounds = atoi(buf);
	return(bounds);

    } else {

	if(conf->dir == NULL){
	    if  ((bfname=(char *)malloc(10)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(bfname, "bounds");
	}else{
	    if  ((bfname=(char *)malloc(strlen(conf->dir) + 10)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(bfname, "%s/bounds", conf->dir);
	}

#ifdef DEBUG
	printf("bounds file name: %s\n", bfname);
#endif
	if ((boundsf = open(bfname, O_RDONLY, 0)) == -1) {
	    perror("Opening bounds file for read");
	    fprintf(stderr, "Will create bounds file\n");
	    free(bfname);
	    return -1;
	}
	// shamelessly reuses bfname as buffer for ascii bounds value

	if (read(boundsf, bfname, strlen(bfname)) == -1) {
	    perror("Could not read bounds");
	    fprintf(stderr, "Will create bounds file\n");
	    free(bfname);
	    close(boundsf);
	    return -1;
	}
	bounds = atoi(bfname);
	close(boundsf);
	free(bfname);
	return (bounds);
    }
}

int
check_dev( char * dumpdev)
{
    struct stat statbuf;	

    // check input file: dumpdev

    if (stat(dumpdev, &statbuf) == -1) {
	perror("Can't stat dumpdev");
	exit(40);
    }

    if (!(S_ISBLK(statbuf.st_mode))) {
	fprintf(stderr,"%s is not a block device\n", dumpdev);
	exit(41);
    }
#ifdef DEBUG
    else {
	fprintf(stderr, "%s IS a block device\n", dumpdev);
    }
#endif
    return 0;
}

int
copy_dump(struct confInfo *conf, int bounds)
{ 
    char *dfname, *pfname, *pagebuf, *headerbuf, *fuser, *fpass, sbounds[10];
    FILE *dumpf;
    int  devicef, ret;
    int dump_header_offset, dump_header_size;
    uint64_t dha_magic_number[64]; 

    dump_page_t dp;
    generic_dump_header_t *dh;
    generic_dump_header_asm_t *dha;
    uint32_t page_index = 0;
    int page_nbr = 0, err = 0;


    // set up filename for output file, and open it up
    //
    if(conf->ftp){

	if(conf->path == NULL){
	    if  ((dfname=(char *)malloc(20)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    
	    if  ((pfname=(char *)malloc(20)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(dfname, "%d", bounds);
	    sprintf(pfname, "%d/dump.%d", bounds,bounds);
	}else{
	    if  ((dfname=(char *)malloc(strlen(conf->path) + 20)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    if  ((pfname=(char *)malloc(strlen(conf->path) + 20)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	    }
	    sprintf(dfname, "%s/%d", conf->path, bounds);
	    sprintf(pfname, "%s/%d/dump.%d", conf->path, bounds,bounds);
	}

	if  ((fuser=(char *)malloc(strlen(conf->user)+7)) == NULL){
	    perror("FATAL ERROR: Couldn't allocate enough memory");
	    exit(10);
	}

	if  ((fpass=(char *)malloc(strlen(conf->pass)+7)) == NULL){
	    perror("FATAL ERROR: Couldn't allocate enough memory");
	    exit(10);
	}

	sprintf(fuser, "USER %s", conf->user);
	sprintf(fpass, "PASS %s", conf->pass);

	if ( ((err = FTPConnect(conf->host, fuser, fpass)) != 0) ){
	    fprintf(stderr, "Could not connect to server!\n");
	    free(fuser);
	    free(fpass);
	    free(dfname);
	    free(pfname);
	    exit(err);
	}

	if(do_mkdir(dfname) !=0)
	    fprintf(stderr, "Could not create directory, it may already exsist.!\n");



	if ( (dumpf = do_open(pfname)) == NULL ){
	    fprintf(stderr, "Could not create file, it may already exsist, or you don't have permission!\n");
	    free(fuser);
	    free(fpass);
	    free(dfname);
	    free(pfname);
	    disconnect();
	    return 39;
	}

	free(fuser);
	free(fpass);

    }else{
	if  ((dfname=(char *)malloc(strlen(conf->dir) + 20)) == NULL){
	    perror("FATAL ERROR: Couldn't allocate enough memory");
	    exit(10);
	}
	sprintf(dfname, "%s/%d", conf->dir, bounds);


	if (!direxists(dfname))
	    if(!mkdir(dfname, 0755)){
	    }

	sprintf(dfname, "%s/%d/dump.%d", conf->dir, bounds, bounds);

	//	if ((dumpf = open(dfname, O_CREAT|O_RDWR|O_TRUNC, 0644)) == -1) 
	if ((dumpf = fopen(dfname, "w")) == NULL){
	    perror("open of dump file");
	    free(dfname);
	    return 81;
	}	
    }

#ifdef DEBUG
    fprintf(stderr,"output filename: %s\n", dfname);
    fprintf(stderr,"dumpfile fs: %d\n", dumpf);
#endif

    free(dfname);

    // Open device partition with dump on it
    //

    if((devicef = open(conf->dev, O_RDONLY, 0)) == -1)
    {
	perror("Open dumpdev");
	return 83;
    }
#ifdef DEBUG
    fprintf(stderr,"dumpdev fs: %d\n", devicef);
#endif

    // fixed offsets:
    // DUMP_HEADER_OFFSET is the start of the headers on the raw device.a
    // DUMP_HEADER_SIZE is the size of all headers plus pad before the
    //    page dumps start.
    //

    if(kver() == 4)
	    dump_header_offset = 4 * 1024;
    else 
	    dump_header_offset = DUMP_HEADER_OFFSET;

    dump_header_size = DUMP_HEADER_SIZE;

    // Move beyond swap header
    //

#ifdef DEBUG
    fprintf(stderr,"just before seek, dho=%d\n", dump_header_offset);
#endif
    if (lseek(devicef, dump_header_offset, SEEK_SET) == -1) {
	perror("lseek on dumpdev");
	close(devicef);
	return 84;
    }

    //  read the whole dump header block (why not?)
    //

    headerbuf=(char *)malloc(dump_header_size);
    if (headerbuf == NULL) {
	perror("malloc dump header");
	exit(10);
    }

    if (read(devicef, headerbuf, dump_header_size) <  dump_header_size) {
	perror("read dump headers");
	free(headerbuf);
	return 84;
    }

    //  check the generic parts of the dump header
    //

    dh = (generic_dump_header_t *)headerbuf;

#ifdef DEBUG
    fprintf(stderr," dh_offset = %d, dh_size = %d, page_size = %d\n",
	    dump_header_offset, dh->dh_header_size, 
	    dh->dh_dump_page_size);
#endif

    // Validate magic number
    //

    if (dh->dh_magic_number != DUMP_MAGIC_NUMBER) {
	fprintf(stderr, "No dump to save\n");
#ifdef DEBUG		
	printf("%llx != %llx\n", dh->dh_magic_number, 
		DUMP_MAGIC_NUMBER);
#endif
	close(devicef);
	return 85;
    }

    // Find the asm_dump_header
    //

    // use memcpy instead of dha = (..._t *)(headerbuf + dh->dh_header_size)
    // to avoide unaligned access. 
    memcpy(dha_magic_number, headerbuf + dh->dh_header_size, sizeof(dha_magic_number));

#ifdef DEBUG
    fprintf(stderr,"dump header asm size = %d\n", dha->dha_header_size);
#endif

    // Validate asm magic number
    // 

    if (*dha_magic_number != DUMP_ASM_MAGIC_NUMBER) {
	fprintf(stderr, "BAD MAGIC Asm dump header invalid!\n");
#ifdef DEBUG
	fprintf(stderr,"%llx != %llx\n", dha->dha_magic_number, 
		DUMP_ASM_MAGIC_NUMBER);
#endif
	close(devicef);
	return 85;
    }


    //  write the dump header block
    //    optionally write just the two headers, and seek to
    //    dump_header_size
    //
    //  if (write(dumpf, headerbuf, dump_header_size) != dump_header_size) 
    if ((ret = fwrite(headerbuf, 1, dump_header_size, dumpf)) < dump_header_size) {
	perror("write dump header");
	close(devicef);
	if(conf->ftp)
	    disconnect();
	else
	    fclose(dumpf);
	return 86;
    }

    //   check if just a header, no pages:
    //  

    if (dh->dh_dump_level == DUMP_LEVEL_HEADER)
    {
	fprintf(stderr, "Dump is nothing but header!\n");
	close(devicef);
	if(conf->ftp)
	    disconnect();
	else
	    fclose(dumpf);
	sprintf(sbounds, "%d", bounds+1);
	createbounds(conf, sbounds);
	erase_dump(conf->dev);
	return 87;
    }


    //  Seek past swap and dump headers in input, dump header in output
    //

    (void)lseek(devicef, dump_header_offset + dump_header_size, SEEK_SET);
    (void)fseek(dumpf, dump_header_size, SEEK_SET);

    //  loop to write pages
    //

    pagebuf=(char *)malloc(dh->dh_dump_page_size);
    page_index=0;

    while (1)
    {
	// read a page header, then the page (below)
	//

	if (read(devicef, (char *)&dp, sizeof(dp)) != sizeof(dp))
	{
	    perror("\nReading dump page header");
	    if(conf->ftp)
		disconnect();
	    else
		fclose(dumpf);
	    free(pagebuf);
	    close(devicef);
	    sprintf(sbounds, "%d", bounds+1);
	    createbounds(conf, sbounds);
	    return 88;
	}


	// some sanity checks:
	//

	if ((dp.dp_flags == 0) || (dp.dp_flags >= DUMP_DH_NOT_USED) ||
		((dp.dp_size == 0) && (!(dp.dp_flags & DUMP_DH_END))) )  {
	    fprintf(stderr, "\ndump page flags wrong! " "Maybe truncated\n");
#ifdef DEBUG
	    fprintf(stderr,
		    "dp.dp_flags = 0x%x\ndp.dp_size=0x%x\n"
		    "dp.dp_address=0x%lx, page_index=%d\n", 
		    dp.dp_flags, dp.dp_size,
		    dp.dp_address, page_index);
#endif			

	    if(conf->ftp)
		disconnect();
	    else
		fclose(dumpf);
	    free(pagebuf);
	    free(dh);
	    close(devicef);
	    sprintf(sbounds, "%d", bounds+1);
	    createbounds(conf, sbounds);
	    erase_dump(conf->dev);
	    return 88;
	}
	// Check that page size isn't too big
	//

	if (dp.dp_size > dh->dh_dump_page_size) {
	    fprintf(stderr, "\npage size too big! May be corrupt.\n");
#ifdef DEBUG
	    fprintf(stderr,
		    "\ndp.dp_size =%u\n, dh->dh_dump_page_size =%u\n",
		    dp.dp_size, dh->dh_dump_page_size);
#endif			
	    if(conf->ftp)
		disconnect();
	    else
		fclose(dumpf);
	    free(dh);
	    free(pagebuf);
	    close(devicef);
	    sprintf(sbounds, "%d", bounds+1);
	    createbounds(conf, sbounds);
	    erase_dump(conf->dev);
	    return 88;
	}

	// write dump page header
	//

	//	       write(dumpf, (char *)&dp, sizeof(dp)) != sizeof(dp) 
	if ( (ret = fwrite((char *)&dp, 1, sizeof(dp), dumpf)) < sizeof(dp)) {
	    perror("\nWrite dump page header");
	    if(conf->ftp)
		disconnect();
	    else
		fclose(dumpf);
	    free(dh);
	    free(pagebuf);
	    close(devicef);
	    sprintf(sbounds, "%d", bounds+1);
	    createbounds(conf, sbounds);
	    return 86;
	}

	//  Check for end of dump.
	//

	if (dp.dp_flags & DUMP_DH_END) {
	    percent_done(dh->dh_num_dump_pages,
		    dh->dh_num_dump_pages);

	    // "erase" the dump header by changing the magic
	    // number

	    close(devicef);
	    free(dh);
	    free(pagebuf);
	    sprintf(sbounds, "%d", bounds+1);
	    createbounds(conf, sbounds);
	    if(conf->ftp)
		disconnect();
	    else
		fclose(dumpf);
	    
	    erase_dump(conf->dev);
	    if (dp.dp_flags & DUMP_DH_TRUNCATED) {
		fprintf(stderr,"\nDump truncated\n");
		return 89;
	    } else{

		return 0;
	    }
	}

	//  read in actual page, write it out.
	//

	if (read(devicef, pagebuf, dp.dp_size) != dp.dp_size) {
	    perror("\nReading page data");
	    if(conf->ftp)
		disconnect();
	    else
		fclose(dumpf);
	    close(devicef);
	    free(dh);
	    free(pagebuf);
	    sprintf(sbounds, "%d", bounds+1);
	    createbounds(conf, sbounds);
	    return 84;
	}

	//	   write(dumpf, (char *)pagebuf, dp.dp_size) != dp.dp_size) 
	if ((ret = fwrite((char *)pagebuf, 1, dp.dp_size, dumpf)) != dp.dp_size) {
	    perror("\nWriting page data");
	    if(conf->ftp)
		disconnect();
	    else
		fclose(dumpf);
	    close(devicef);
	    free(dh);
	    free(pagebuf);
	    sprintf(sbounds, "%d", bounds+1);
	    createbounds(conf, sbounds);
	    return 86;
	}

	//  update percentage
	//

	percent_done(page_nbr++, dh->dh_num_dump_pages);
	page_index++;

    } // end while

    // never get here
    //

} // end copy dump

int 
reset_dump(char *ddev)
{
    int fs;
    int offset;
    generic_dump_header_t dhead;

    if (kver() == 4)
	    offset = 4 * 1024;
    else 
	    offset = DUMP_HEADER_OFFSET;
    // check ddev, is it block dev?
    //

    check_dev(ddev);


    if ((fs=open(ddev,O_RDWR, 0)) == -1) {
	perror("reset:open");
	return 83;
    }

#ifdef DEBUG
    fprintf(stderr, "offset=%u, fs=%d, ddev=%s\n", offset, fs, ddev);
#endif

    if (lseek(fs, offset, SEEK_SET) != offset) {
	perror("reset:lseek");
	return 84;
    }

    if (read(fs, (char *)&dhead, sizeof(dhead)) != 
	    (ssize_t)sizeof(dhead)) {
	perror("reset: read");
	return 84;
    }

    // Check to try to ascertain if we're looking at the right thing...
    //

#ifdef DEBUG
    fprintf(stderr,"dhead.dh_magic_number = %llx\n", dhead.dh_magic_number);
#endif

    if (dhead.dh_magic_number == DUMP_MAGIC_NUMBER) {
	fprintf(stderr, "dump is already good\n");
	close(fs);
	return 0;
    } else if (dhead.dh_magic_number == DUMP_MAGIC_ERASED) {
	dhead.dh_magic_number = DUMP_MAGIC_NUMBER;

	if (lseek(fs, offset, SEEK_SET) != offset) {
	    perror("reset:lseek back");
	    return 84;
	}

	if (write(fs,(char *)&dhead, sizeof(dhead)) < 
		(ssize_t)sizeof(dhead)) {
	    perror("reset: write");
	    return 86;
	}
	fprintf(stderr,"dump reset to good\n");
	close(fs);
	return 0;
    } else {
	fprintf(stderr,"Not a dump magic number! Got dump device?\n");
	close(fs);
	return 85;
    }

}  // end reset_dump

int
erase_dump(char *ddev)
{
    int fs;
    int offset;
    generic_dump_header_t dhead;
    
    if (kver() == 4)
	    offset = 4 * 1024;
    else 
	    offset = DUMP_HEADER_OFFSET;

    // check ddev, is it block dev?
    //

    check_dev(ddev);

    if ((fs=open(ddev,O_RDWR, 0)) == -1) {
	perror("erase: open");
	return 83;
    }

#ifdef DEBUG
    fprintf(stderr, "offset=%u, fs=%d, ddev=%s\n", offset, fs, ddev);
#endif

    if (lseek(fs, offset, SEEK_SET) != offset) {
	perror("erase:lseek");
	return 84;
    }

    if (read(fs, (char *)&dhead, sizeof(dhead)) != sizeof(dhead)) {
	perror("erase: read");
	return 84;
    }

    // Check to try to ascertain if we're looking at the right thing...
    //

#ifdef DEBUG
    fprintf(stderr,"erase: dhead.dh_magic_number = %llx\n", 
	    dhead.dh_magic_number);
    fprintf(stderr,"erase: mn = %llx\n", DUMP_MAGIC_NUMBER);
#endif

    if (dhead.dh_magic_number == DUMP_MAGIC_NUMBER) {
	dhead.dh_magic_number = DUMP_MAGIC_ERASED;

	if (lseek(fs, offset, SEEK_SET) != offset) {
	    perror("erase:lseek back");
	    return 84;
	}

	if (write(fs,(char *)&dhead, sizeof(dhead)) < 
		(ssize_t)sizeof(dhead)) {
	    perror("erase: write");
	    return 86;
	}
#ifdef DEBUG
	fprintf(stderr,"Dump erased?\n");
	if (lseek(fs, offset, SEEK_SET) != offset) {
	    perror("erase:lseek back");
	    return 84;
	}

	if (read(fs, (char *)&dhead, sizeof(dhead)) != sizeof(dhead)) {
	    perror("erase: read");
	    return 84;
	}
	fprintf(stderr, "dhead.dh_magic_number = %llx on disk\n",
		dhead.dh_magic_number);
#endif
	close(fs);
	return 0;
    } else if (dhead.dh_magic_number == DUMP_MAGIC_ERASED) {
	fprintf(stderr,"dump already erased\n");
	close(fs);
	return 0;
    } else {
	fprintf(stderr,"not a dump\n");
	close(fs);
	return 85;
    }

}  // end erase_dump

struct confInfo *
parseConf(struct confInfo *conf)
{
    FILE *fp;
    int i, active, save=0;
    char line[MAX_LINE_SIZE];
    char *tok, *kill_nl;
    char config[] = "/etc/dumputils.conf";
    struct pconf *parse;

    if  ((parse=malloc(sizeof(struct pconf))) == NULL){
	perror("FATAL ERROR: Couldn't allocate enough memory");
	exit(10);
    }


    if(conf->error != NULL){
	free(conf->error);
	conf->error == NULL;
    }

    parse = parseconf(config);
    
    if (parse == NULL) {
	fprintf(stderr, "ERROR: Can not find /etc/dumputils.conf\n");
	exit(53);
    }

    if (!parse->dumpdev[0]) {
	fprintf(stderr, "FATAL ERROR: no valid DUMPDEV= parameter found in: %s\n", config);
	exit(50);
    }else
	conf->dev = strdup(parse->dumpdev);

    if (!parse->dumpdir[0]) {
	fprintf(stderr, "FATAL ERROR: no valid DUMPDIR= parameter found in: %s\n", config);
	exit(51);
    }else
	conf->dir = strdup(parse->dumpdir);
    
    if (!parse->dump_save[0]) {
	fprintf(stderr, "FATAL ERROR: no valid DUMP_SAVE= parameter found in: %s\n", config);
	exit(52);
    }else
	save = atoi(parse->dump_save);


    free(parse);

    if (save == 0){
	conf->error=strdup("DUMP_SAVE = 0 in config file. Will not save dump");
	return conf;
    }

    if((tok=strstr(conf->dir,"ftp://"))){
	conf->ftp = 1;

	if( (i=(parseUrl(conf->dir, &conf->user, &conf->pass, 
			    &conf->host, &conf->port, &conf->path)) != 0)){
	    conf->error=strdup("ERROR parsing ftp URL");
	    return conf;
	}
    }

    return conf;
}

int
main(int argc, char **argv)
{
    char *rvalue=NULL, *evalue=NULL, *ovalue=NULL, *dvalue=NULL;
    char *dumpdev, *dumpdir, *tok;
    int c, i, bounds;;
    struct stat statbuf;
    struct confInfo *conf;
    int option = 0;
    char sbounds[10];

    if  ((conf=malloc(sizeof(struct confInfo))) == NULL){
	perror("FATAL ERROR: Couldn't allocate enough memory");
	exit(10);
    }
    conf->dev = NULL;
    conf->dir = NULL;
    conf->user = NULL;
    conf->pass = NULL;
    conf->ftp = 0;
    conf->error = NULL;
    conf->host = NULL;
    conf->path = NULL;
    conf->port = 21;


    // Check that user is root for security
    //
    if (getuid() != 0) {
	fprintf(stderr, "Must xbe root to run this program.\n");
	exit(40);
    }


    opterr = 0;

    while ((c = getopt (argc, argv, "v?r:e:d:o:")) != -1)
	switch (c) {
	    case 'd':
		dvalue = optarg;
		option += 1;
		break;
	    case 'e':
		if ((erase_dump(optarg)) < 0) {
		    fprintf(stderr,"erase dump failed\n");
		    exit(61);
		} else 
		    fprintf(stderr, "dump is erased\n");
		return 0;
	    case 'o':
		ovalue = optarg;
		option += 10;
		break;
	    case 'r':
		if ((reset_dump(optarg)) < 0) {
		    fprintf(stderr,"reset dump failed\n");
		    exit(62);
		}
		return 0;
	    case '?':
		usage(argv[0]);
		exit(0);
	    case 'v':
		version(argv[0]);
		exit(0);
	    default:
		fprintf(stderr, "Unknown Error, comand parsing in trouble\n");
		exit(63);
	}


    switch (option) {
	// Default, read everything from config file.
	case 0:
	    parseConf(conf);
	    break;

	    // User override of dump device via cmd line
	case 1:
	    parseConf(conf);
	    if(conf->dev != NULL)
		free(conf->dev);
	    conf->dev = dvalue;
	    break;


	    // User override of output directory via cmd line 
	case 10:
	    parseConf(conf);

	    if(conf->user != NULL)
		free(conf->user);
	    if(conf->pass != NULL)
		free(conf->pass);
	    if(conf->host != NULL)
		free(conf->host);
	    if(conf->path != NULL)
		free(conf->path);

	    conf->user = NULL;
	    conf->pass = NULL;
	    conf->host = NULL;
	    conf->path = NULL;
	    conf->port = 21;

	    if((tok=strstr(ovalue,"ftp://"))){
		conf->ftp = 1;

		if( (i=(parseUrl(ovalue, &conf->user, &conf->pass, 
				    &conf->host, &conf->port, &conf->path)) != 0)){
		    fprintf(stderr, "ERROR parsing ftp URL\n");
		    exit(i);
		}

	    }else{
		if(conf->dir != NULL)
		    free(conf->dir);
		conf->ftp = 0;
		conf->dir = ovalue;
	    }
	    break;

	    // User override of output and dump dev via cmd line. 
	case 11:
	    parseConf(conf);

	    if(conf->user != NULL)
		free(conf->user);
	    if(conf->pass != NULL)
		free(conf->pass);
	    if(conf->host != NULL)
		free(conf->host);
	    if(conf->path != NULL)
		free(conf->path);

	    conf->user = NULL;
	    conf->pass = NULL;
	    conf->host = NULL;
	    conf->path = NULL;
	    conf->port = 21;

	    if((tok=strstr(ovalue,"ftp://"))){
		conf->ftp = 1;

		if( (i=(parseUrl(ovalue, &conf->user, &conf->pass, 
				    &conf->host, &conf->port, &conf->path)) != 0)){
		    fprintf(stderr, "ERROR parsing ftp URL\n");
		    exit(i);
		}

	    }else{
		if(conf->dir != NULL)
		    free(conf->dir);
		conf->ftp = 0;
		conf->dir = ovalue;
	    }

	    if(conf->dev != NULL)
		free(conf->dev);
	    conf->dev = dvalue;

	    break;
    }


    if(conf->ftp){
	if(conf->user == NULL)
	    conf->user = strdup("anonymous");
	if(conf->pass == NULL)
	    conf->pass = strdup("anonymous@anonymous.com");
    }else{
	// check output directory: dumpdir
	//
	if (stat(conf->dir, &statbuf) == -1) {
	    perror("Can't stat dumpdir");
	    exit(64);
	}

	if (!(S_ISDIR(statbuf.st_mode))) {
	    fprintf(stderr,"%s is not a directory\n", conf->dir);
	    exit(65);
	}
    }

#ifdef DEBUG
    printf("DEV: %s\n",conf->dev);
    printf("DIR: %s\n",conf->dir);
    printf("FTP: %d\n",conf->ftp);
    printf("HOST: %s\n",conf->host);
    printf("USER: %s\n",conf->user);
    printf("PASS: %s\n",conf->pass);
    printf("PATH: %s\n",conf->path);
    printf("PORT: %d\n",conf->port);
    printf("ERROR: %s\n",conf->error);
#endif

    // Check input device partition
    //
    check_dev(conf->dev);

    if ((bounds=getbounds(conf)) < 0){ // If bounds does not exsist 
	if ((i = createbounds(conf, "0")) != 0){ // Create it.
	    fprintf(stderr, "FATAL ERROR: Can not read or create bounds file: %d\n", i);
	    return i;
	}else
	    bounds=0;
    }

#ifdef DEBUG
    printf("bounds : %d\n",bounds);
#endif 

    printf("Copying dump...\n");
    if ( (i=copy_dump(conf, bounds)) != 0) {
	fprintf(stderr, "Error while copying dump\n");
	exit (i);
    }

    if(i == 50) // No dump to save
	return 0;

    printf("Successfully wrote dump\n");

    if ( (i=copy_map(conf, bounds)) < 0) {
	fprintf(stderr, "Error while copying System Map\n");
	exit (i);
    }

    printf("Successfully wrote System.map\n");

    return 0;
} //end main 
