package DBIShell::help::mysql;

use strict;
use Exporter ();

use vars qw($VERSION @EXPORT @EXPORT_OK %EXPORT_TAGS @ISA %HELP);

@ISA         = qw(Exporter);
@EXPORT      = ();
@EXPORT_OK   = ();
%EXPORT_TAGS = ();
$VERSION     = 0.01_02;

use constant H_ARITHMETIC => <<'__arithmetic__';
Arithmetic:

The usual arithmetic operators are available.  Note that in the case of
`-', `+' and `*', the result is calculated with `BIGINT' (64-bit)
precision if both arguments are integers!

`+'
     Addition
          mysql> select 3+5;
                  -> 8

`-'
     Subtraction
          mysql> select 3-5;
                  -> -2

`*'
     Multiplication
          mysql> select 3*5;
                  -> 15
          mysql> select 18014398509481984*18014398509481984.0;
                  -> 324518553658426726783156020576256.0
          mysql> select 18014398509481984*18014398509481984;
                  -> 0

     The result of the last expression is incorrect because the result
     of the integer multiplication exceeds the 64-bit range of `BIGINT'
     calculations.

`/'
     Division
          mysql> select 3/5;
                  -> 0.60

     Division by zero produces a `NULL' result:

          mysql> select 102/(1-1);
                  -> NULL

     A division will be calculated with `BIGINT' arithmetic only if
     performed in a context where its result is converted to an integer!

`%'
     Modulo (like the `%' operator in C).  Returns the remainder of `N'
     divided by `M'.
          mysql> select MOD(234, 10);
                  -> 4
          mysql> select 253 % 7;
                  -> 1
          mysql> select MOD(29,9);
                  -> 2

     This function is safe to use with `BIGINT' values.

__arithmetic__

use constant H_BIT_OPERATORS => <<'__bit_operators__';
Bit Operators:

*MySQL* uses `BIGINT' (64-bit) arithmetic for bit operations, so these
operators have a maximum range of 64 bits.

`|'
     Bitwise OR
          mysql> select 29 | 15;
                  -> 31

`&'
     Bitwise AND
          mysql> select 29 & 15;
                  -> 13

`<<'
     Shifts a longlong (`BIGINT') number to the left.
          mysql> select 1 << 2
                  -> 4

`>>'
     Shifts a longlong (`BIGINT') number to the right.
          mysql> select 4 >> 2
                  -> 1

`~'
     Invert all bits.
          mysql> select 5 & ~1
                  -> 4

`BIT_COUNT(N)'
     Returns the number of bits that are set in the argument `N'.
          mysql> select BIT_COUNT(29);
                  -> 4

__bit_operators__

use constant H_LOGICAL_OPERATORS => <<'__logical_operators__';
Logical Operators:

All logical functions return `1' (TRUE) or `0' (FALSE).

`NOT'
`!'
     Logical NOT. Returns `1' if the argument is `0', otherwise returns
     `0'.  Exception: `NOT NULL' returns `NULL'.
          mysql> select NOT 1;
                  -> 0
          mysql> select NOT NULL;
                  -> NULL
          mysql> select ! (1+1);
                  -> 0
          mysql> select ! 1+1;
                  -> 1
     The last example returns `1' because the expression evaluates the
     same way as `(!1)+1'.

`OR'
`||'
     Logical OR. Returns `1' if either argument is not `0' and not
     `NULL'.
          mysql> select 1 || 0;
                  -> 1
          mysql> select 0 || 0;
                  -> 0
          mysql> select 1 || NULL;
                  -> 1

`AND'
`&&'
     Logical AND. Returns `0' if either argument is `0' or `NULL',
     otherwise returns `1'.
          mysql> select 1 && NULL;
                  -> 0
          mysql> select 1 && 0;
                  -> 0

__logical_operators__

use constant H_COMPARISON => <<'__comparison__';
Comparison Operators:

Comparison operations result in a value of `1' (TRUE), `0' (FALSE) or
`NULL'. These functions work for both numbers and strings.  Strings are
automatically converted to numbers and numbers to strings as needed (as
in Perl).

*MySQL* performs comparisons using the following rules:

   * If one or both arguments are `NULL', the result of the comparison
     is `NULL', except for the `<=>' operator.

   * If both arguments in a comparison operation are strings, they are
     compared as strings.

   * If both arguments are integers, they are compared as integers.

   * Hexadecimal values are treated as binary strings if not compared
     to a number.

   * If one of the arguments is a `TIMESTAMP' or `DATETIME' column and
     the other argument is a constant, the constant is converted to a
     timestamp before the comparison is performed. This is done to be
     more ODBC-friendly.

   * In all other cases, the arguments are compared as floating-point
     (real) numbers.

By default, string comparisons are done in case-independent fashion
using the current character set (ISO-8859-1 Latin1 by default, which
also works excellently for English).

The examples below illustrate conversion of strings to numbers for
comparison operations:

     mysql> SELECT 1 > '6x';
              -> 0
     mysql> SELECT 7 > '6x';
              -> 1
     mysql> SELECT 0 > 'x6';
              -> 0
     mysql> SELECT 0 = 'x6';
              -> 1

`='
     Equal
          mysql> select 1 = 0;
                  -> 0
          mysql> select '0' = 0;
                  -> 1
          mysql> select '0.0' = 0;
                  -> 1
          mysql> select '0.01' = 0;
                  -> 0
          mysql> select '.01' = 0.01;
                  -> 1

`<>'
`!='
     Not equal
          mysql> select '.01' <> '0.01';
                  -> 1
          mysql> select .01 <> '0.01';
                  -> 0
          mysql> select 'zapp' <> 'zappp';
                  -> 1

`<='
     Less than or equal
          mysql> select 0.1 <= 2;
                  -> 1

`<'
     Less than
          mysql> select 2 <= 2;
                  -> 1

`>='
     Greater than or equal
          mysql> select 2 >= 2;
                  -> 1

`>'
     Greater than
          mysql> select 2 > 2;
                  -> 0

`<=>'
     Null safe equal
          mysql> select 1 <=> 1, NULL <=> NULL, 1 <=> NULL;
                  -> 1 1 0

`IS NULL'
`IS NOT NULL'
     Test whether or not a value is or is not `NULL'
          mysql> select 1 IS NULL, 0 IS NULL, NULL IS NULL:
                  -> 0 0 1
          mysql> select 1 IS NOT NULL, 0 IS NOT NULL, NULL IS NOT NULL;
                  -> 1 1 0

`expr BETWEEN min AND max'
     If `expr' is greater than or equal to `min' and `expr' is less
     than or equal to `max', `BETWEEN' returns `1', otherwise it
     returns `0'.  This is equivalent to the expression `(min <= expr
     AND expr <= max)' if all the arguments are of the same type.  The
     first argument (`expr') determines how the comparison is performed
     as follows:

        * If `expr' is a `TIMESTAMP', `DATE' or `DATETIME' column, min
          and max are formatted to the same format if they are
          constants.

        * If `expr' is a case-insensitive string expression, a
          case-insensitive string comparison is done.

        * If `expr' is a case-sensitive string expression, a
          case-sensitive string comparison is done.

        * If `expr' is an integer expression, an integer comparison is
          done.

        * Otherwise, a floating-point (real) comparison is done.

          mysql> select 1 BETWEEN 2 AND 3;
                  -> 0
          mysql> select 'b' BETWEEN 'a' AND 'c';
                  -> 1
          mysql> select 2 BETWEEN 2 AND '3';
                  -> 1
          mysql> select 2 BETWEEN 2 AND 'x-3';
                  -> 0

`expr IN (value,...)'
     Returns `1' if `expr' is any of the values in the `IN' list, else
     returns `0'.  If all values are constants, then all values are
     evaluated according to the type of `expr' and sorted. The search
     for the item is then done using a binary search. This means `IN'
     is very quick if the `IN' value list consists entirely of
     constants.  If `expr' is a case-sensitive string expression, the
     string comparison is performed in case-sensitive fashion.

          mysql> select 2 IN (0,3,5,'wefwf');
                  -> 0
          mysql> select 'wefwf' IN (0,3,5,'wefwf');
                  -> 1

`expr NOT IN (value,...)'
     Same as `NOT (expr IN (value,...))'.

see also: string_comparison, isnull, coalesce, interval
__comparison__


use constant H_ISNULL => <<'__isnull__';
ISNULL(expr):

     If `expr' is `NULL', `ISNULL()' returns `1', otherwise it returns
     `0'.
          mysql> select ISNULL(1+1);
                  -> 0
          mysql> select ISNULL(1/0);
                  -> 1

     Note that a comparison of `NULL' values using `=' will always be
     false!

__isnull__

use constant H_COALESCE => <<'__coalesce__';
COALESCE(list):

     Returns first non-`NULL' element in list.

          mysql> select COALESCE(NULL,1);
                  -> 1
          mysql> select COALESCE(NULL,NULL,NULL);
                  -> NULL

__coalesce__

use constant H_INTERVAL => <<'__interval__';
INTERVAL(N,N1,N2,N3,...):

     Returns `0' if `N' < `N1', `1' if `N' < `N2' and so on. All
     arguments are treated as integers.  It is required that `N1' <
     `N2' < `N3' < `...' < `Nn' for this function to work correctly.
     This is because a binary search is used (very fast).

          mysql> select INTERVAL(23, 1, 15, 17, 30, 44, 200);
                  -> 3
          mysql> select INTERVAL(10, 1, 10, 100, 1000);
                  -> 2
          mysql> select INTERVAL(22, 23, 30, 44, 200);
                  -> 0

__interval__

use constant H_STRING_COMPARISON => <<'__string_comparison__';

Normally, if any expression in a string comparison is case sensitive,
the comparison is performed in case-sensitive fashion.

`expr LIKE pat [ESCAPE 'escape-char']'
     Pattern matching using SQL simple regular expression comparison.
     Returns `1' (TRUE) or `0' (FALSE).  With `LIKE' you can use the
     following two wildcard characters in the pattern:

     `%'     Matches any number of characters, even zero characters
     `_'     Matches exactly one character

          mysql> select 'David!' LIKE 'David_';
                  -> 1
          mysql> select 'David!' LIKE '%D%v%';
                  -> 1

     To test for literal instances of a wildcard character, precede the
     character with the escape character.  If you don't specify the
     `ESCAPE' character, `\' is assumed:

     `\%'    Matches one `%' character
     `\_'    Matches one `_' character

          mysql> select 'David!' LIKE 'David\_';
                  -> 0
          mysql> select 'David_' LIKE 'David\_';
                  -> 1

     To specify a different escape character, use the `ESCAPE' clause:

          mysql> select 'David_' LIKE 'David|_' ESCAPE '|';
                  -> 1

     `LIKE' is allowed on numeric expressions! (This is a *MySQL*
     extension to the ANSI SQL `LIKE'.)

          mysql> select 10 LIKE '1%';
                  -> 1

     Note: Because *MySQL* uses the C escape syntax in strings (e.g.,
     `\n'), you must double any `\' that you use in your `LIKE'
     strings.  For example, to search for `\n', specify it as `\\n'.  To
     search for `\', specify it as `\\\\' (the backslashes are stripped
     once by the parser, and another time when the pattern match is
     done, leaving a single backslash to be matched).

`expr NOT LIKE pat [ESCAPE 'escape-char']'
     Same as `NOT (expr LIKE pat [ESCAPE 'escape-char'])'.

`expr REGEXP pat'
`expr RLIKE pat'
     Performs a pattern match of a string expression `expr' against a
     pattern `pat'.  The pattern can be an extended regular expression.
     *Note Regexp::.  Returns `1' if `expr' matches `pat', otherwise
     returns `0'.  `RLIKE' is a synonym for `REGEXP', provided for
     `mSQL' compatibility. Note: Because *MySQL* uses the C escape
     syntax in strings (e.g., `\n'), you must double any `\' that you
     use in your `REGEXP' strings.  In `MySQL' 3.23.4 `REGEXP' is case
     insensitive for normal (not binary) strings.

          mysql> select 'Monty!' REGEXP 'm%y%%';
                  -> 0
          mysql> select 'Monty!' REGEXP '.*';
                  -> 1
          mysql> select 'new*\n*line' REGEXP 'new\\*.\\*line';
                  -> 1
          mysql> select "a" REGEXP "A", "a" REGEXP BINARY "A";
                  -> 1  0

`'
     `REGEXP' and `RLIKE' use the current character set (ISO-8859-1
     Latin1 by default) when deciding the type of a character.

`expr NOT REGEXP pat'
`expr NOT RLIKE pat'
     Same as `NOT (expr REGEXP pat)'.

see also: comparison, strcmp
__string_comparison__


use constant H_STRCMP => <<'__strcmp__';
STRCMP(expr1,expr2):

     `STRCMP()' returns `0' if the strings are the same, `-1' if the
     first argument is smaller than the second according to the current
     sort order, and `1' otherwise.

          mysql> select STRCMP('text', 'text2');
                  -> -1
          mysql> select STRCMP('text2', 'text');
                  -> 1
          mysql> select STRCMP('text', 'text');
                  -> 0

__strcmp__

use constant H_BINARY => <<'__binary__';
BINARY:

     The `BINARY' operator casts the string following it to a binary
     string.  This is an easy way to force a column comparison to be
     case sensitive even if the column isn't defined as `BINARY' or
     `BLOB'.
          mysql> select "a" = "A";
                  -> 1
          mysql> select BINARY "a" = "A";
                  -> 0

     `BINARY' was introduced in *MySQL* 3.23.0

__binary__


use constant H_IFNULL => <<'__ifnull__';
IFNULL(expr1,expr2):

     If `expr1' is not `NULL', `IFNULL()' returns `expr1', else it
     returns `expr2'.  `IFNULL()' returns a numeric or string value,
     depending on the context in which it is used.
          mysql> select IFNULL(1,0);
                  -> 1
          mysql> select IFNULL(NULL,10);
                  -> 10
          mysql> select IFNULL(1/0,10);
                  -> 10
          mysql> select IFNULL(1/0,'yes');
                  -> 'yes'

__ifnull__

use constant H_NULLIF => <<'__nullif__';
NULLIF(expr1,expr2):

     If `expr1 = expr2' is true, return `NULL' else return `expr1'.
     This is the same as `CASE WHEN x = y THEN NULL ELSE x END'
          mysql> select NULLIF(1,1);
                  -> NULL
          mysql> select NULLIF(1,2);
                  -> 1

     Note that `expr1' is evaluated twice in *MySQL* if the arguments
     are equal.

__nullif__

use constant H_IF => <<'__if__';
IF(expr1,expr2,expr3):

     If `expr1' is TRUE (`expr1 <> 0' and `expr1 <> NULL') then `IF()'
     returns `expr2', else it returns `expr3'.  `IF()' returns a
     numeric or string value, depending on the context in which it is
     used.

          mysql> select IF(1>2,2,3);
                  -> 3
          mysql> select IF(1<2,'yes','no');
                  -> 'yes'
          mysql> select IF(strcmp('test','test1'),'no','yes');
                  -> 'no'

     `expr1' is evaluated as an integer value, which means that if you
     are testing floating-point or string values, you should do so
     using a comparison operation.

          mysql> select IF(0.1,1,0);
                  -> 0
          mysql> select IF(0.1<>0,1,0);
                  -> 1

     In the first case above, `IF(0.1)' returns `0' because `0.1' is
     converted to an integer value, resulting in a test of `IF(0)'.
     This may not be what you expect.  In the second case, the
     comparison tests the original floating-point value to see whether
     it is non-zero.  The result of the comparison is used as an
     integer.

     The default return type of `IF()' (which may matter when it stored
     into a temporary table) is calculated in *MySQL* 3.23 as follows:

     expr2 or expr3 returns string                   string
     expr2 or expr3 returns a floating point value   floating point
     expr2 or expr3 returns an integer               integer

__if__

use constant H_CASE => <<'__case__';
CASE value
   WHEN [compare-value] THEN result
  [WHEN [compare-value] THEN result ...]
  [ELSE result]
END

CASE WHEN [condition] THEN result
    [WHEN [condition] THEN result ...]
    [ELSE result]
END

   The first version returns the `result' where
   `value=compare-value'. The second version returns the result for
   the first condition which is true. If there was no matching result
   value, then the result after `ELSE' is returned. If there is no
   `ELSE' part then `NULL' is returned.

   mysql> SELECT CASE 1 WHEN 1 THEN "one" WHEN 2 THEN "two" ELSE "more" END;
          -> "one"
   mysql> SELECT CASE WHEN 1>0 THEN "true" ELSE "false" END;
          -> "true"
   mysql> SELECT CASE BINARY "B" when "a" then 1 when "b" then 2 END;
          -> NULL

__case__

use constant H_FLOW_CONTROL => <<'__flow_control__';
Flow Control:

See:
    ifnull
    nullif
    if
    case

__flow_control__

use constant H_ABS => <<'__abs__';
ABS(X):

     Returns the absolute value of `X'.
          mysql> select ABS(2);
                  -> 2
          mysql> select ABS(-32);
                  -> 32

     This function is safe to use with `BIGINT' values.

__abs__

use constant H_SIGN => <<'__sign__';
SIGN(X):

     Returns the sign of the argument as `-1', `0' or `1', depending on
     whether `X' is negative, zero, or positive.
          mysql> select SIGN(-32);
                  -> -1
          mysql> select SIGN(0);
                  -> 0
          mysql> select SIGN(234);
                  -> 1

__sign__

use constant H_MOD => <<'__mod__';
MOD(N,M):

     Modulo (like the `%' operator in C).  Returns the remainder of `N'
     divided by `M'.
          mysql> select MOD(234, 10);
                  -> 4
          mysql> select 253 % 7;
                  -> 1
          mysql> select MOD(29,9);
                  -> 2

     This function is safe to use with `BIGINT' values.

__mod__

use constant H_FLOOR => <<'__floor__';
FLOOR(X):

     Returns the largest integer value not greater than `X'.
          mysql> select FLOOR(1.23);
                  -> 1
          mysql> select FLOOR(-1.23);
                  -> -2

     Note that the return value is converted to a `BIGINT'!

__floor__

use constant H_CEILING => <<'__ceiling__';
CEILING(X):

     Returns the smallest integer value not less than `X'.
          mysql> select CEILING(1.23);
                  -> 2
          mysql> select CEILING(-1.23);
                  -> -1

     Note that the return value is converted to a `BIGINT'!

__ceiling__

use constant H_ROUND => <<'__round__';
ROUND(X):

     Returns the argument `X', rounded to the nearest integer.
          mysql> select ROUND(-1.23);
                  -> -1
          mysql> select ROUND(-1.58);
                  -> -2
          mysql> select ROUND(1.58);
                  -> 2

ROUND(X,D):

     Returns the argument `X', rounded to a number with `D' decimals.
     If `D' is `0', the result will have no decimal point or fractional
     part.

          mysql> select ROUND(1.298, 1);
                  -> 1.3
          mysql> select ROUND(1.298, 0);
                  -> 1

__round__

use constant H_EXP => <<'__exp__';
EXP(X):

     Returns the value of `e' (the base of natural logarithms) raised to
     the power of `X'.
          mysql> select EXP(2);
                  -> 7.389056
          mysql> select EXP(-2);
                  -> 0.135335

__exp__

use constant H_LOG => <<'__log__';
LOG(X):

     Returns the natural logarithm of `X'.
          mysql> select LOG(2);
                  -> 0.693147
          mysql> select LOG(-2);
                  -> NULL
     If you want the log of a number `X' to some arbitary base `B', use
     the formula `LOGB(X) = LOG(X)/LOG(B)'.

__log__

use constant H_LOG10 => <<'__log10__';
LOG10(X):

     Returns the base-10 logarithm of `X'.
          mysql> select LOG10(2);
                  -> 0.301030
          mysql> select LOG10(100);
                  -> 2.000000
          mysql> select LOG10(-100);
                  -> NULL

__log10__

use constant H_POW => <<'__pow__';
POW(X,Y):

     Returns the value of `X' raised to the power of `Y'.
          mysql> select POW(2,2);
                  -> 4.000000
          mysql> select POW(2,-2);
                  -> 0.250000

__pow__

use constant H_POWER => H_POW;

use constant H_SQRT => <<'__sqrt__';
SQRT(X):

     Returns the non-negative square root of `X'.
          mysql> select SQRT(4);
                  -> 2.000000
          mysql> select SQRT(20);
                  -> 4.472136

__sqrt__

use constant H_PI => <<'__pi__';
PI():

     Returns the value of PI.
          mysql> select PI();
                  -> 3.141593

__pi__

use constant H_COS => <<'__cos__';
COS(X):

     Returns the cosine of `X', where `X' is given in radians.
          mysql> select COS(PI());
                  -> -1.000000

__cos__

use constant H_SIN => <<'__sin__';
SIN(X):

     Returns the sine of `X', where `X' is given in radians.
          mysql> select SIN(PI());
                  -> 0.000000

__sin__

use constant H_TAN => <<'__tan__';
TAN(X):

     Returns the tangent of `X', where `X' is given in radians.
          mysql> select TAN(PI()+1);
                  -> 1.557408

__tan__

use constant H_ACOS => <<'__acos__';
ACOS(X):

     Returns the arc cosine of `X', that is, the value whose cosine is
     `X'. Returns `NULL' if `X' is not in the range `-1' to `1'.
          mysql> select ACOS(1);
                  -> 0.000000
          mysql> select ACOS(1.0001);
                  -> NULL
          mysql> select ACOS(0);
                  -> 1.570796

__acos__

use constant H_ASIN => <<'__asin__';
ASIN(X):

     Returns the arc sine of `X', that is, the value whose sine is `X'.
     Returns `NULL' if `X' is not in the range `-1' to `1'.
          mysql> select ASIN(0.2);
                  -> 0.201358
          mysql> select ASIN('foo');
                  -> 0.000000

__asin__

use constant H_ATAN => <<'__atan__';
ATAN(X):

     Returns the arc tangent of `X', that is, the value whose tangent is
     `X'.
          mysql> select ATAN(2);
                  -> 1.107149
          mysql> select ATAN(-2);
                  -> -1.107149

__atan__

use constant H_ATAN2 => <<'__atan2__';
ATAN2(X,Y):

     Returns the arc tangent of the two variables `X' and `Y'. It is
     similar to calculating the arc tangent of `Y / X', except that the
     signs of both arguments are used to determine the quadrant of the
     result.
          mysql> select ATAN(-2,2);
                  -> -0.785398
          mysql> select ATAN(PI(),0);
                  -> 1.570796

__atan2__

use constant H_COT => <<'__cot__';
COT(X):

     Returns the cotangent of `X'.
          mysql> select COT(12);
                  -> -1.57267341
          mysql> select COT(0);
                  -> NULL

__cot__

use constant H_RAND => <<'__rand__';
RAND(N):

     Returns a random floating-point value in the range `0' to `1.0'.
     If an integer argument `N' is specified, it is used as the seed
     value.
          mysql> select RAND();
                  -> 0.5925
          mysql> select RAND(20);
                  -> 0.1811
          mysql> select RAND(20);
                  -> 0.1811
          mysql> select RAND();
                  -> 0.2079
          mysql> select RAND();
                  -> 0.7888
     You can't use a column with `RAND()' values in an `ORDER BY'
     clause, because `ORDER BY' would evaluate the column multiple
     times.  In *MySQL* 3.23, you can however do: `SELECT * FROM
     table_name ORDER BY RAND()'

     This is useful to get a random sample of a set `SELECT * FROM
     table1,table2 WHERE a=b AND c<d ORDER BY RAND() LIMIT 1000'.

     Note that a `RAND()' in a `WHERE' clause will be re-evaluated
     every time the `WHERE' is executed.

__rand__

use constant H_LEAST => <<'__least__';
LEAST(X,Y,...):

     With two or more arguments, returns the smallest (minimum-valued)
     argument.  The arguments are compared using the following rules:

        * If the return value is used in an `INTEGER' context, or all
          arguments are integer-valued, they are compared as integers.

        * If the return value is used in a `REAL' context, or all
          arguments are real-valued, they are compared as reals.

        * If any argument is a case-sensitive string, the arguments are
          compared as case-sensitive strings.

        * In other cases, the arguments are compared as
          case-insensitive strings.

          mysql> select LEAST(2,0);
                  -> 0
          mysql> select LEAST(34.0,3.0,5.0,767.0);
                  -> 3.0
          mysql> select LEAST("B","A","C");
                  -> "A"
     In *MySQL* versions prior to 3.22.5, you can use `MIN()' instead
     of `LEAST'.

__least__

use constant H_GREATEST => <<'__greatest__';
GREATEST(X,Y,...):

     Returns the largest (maximum-valued) argument.  The arguments are
     compared using the same rules as for `LEAST'.
          mysql> select GREATEST(2,0);
                  -> 2
          mysql> select GREATEST(34.0,3.0,5.0,767.0);
                  -> 767.0
          mysql> select GREATEST("B","A","C");
                  -> "C"
     In *MySQL* versions prior to 3.22.5, you can use `MAX()' instead
     of `GREATEST'.

__greatest__

use constant H_DEGREES => <<'__degrees__';
DEGREES(X):

     Returns the argument `X', converted from radians to degrees.
          mysql> select DEGREES(PI());
                  -> 180.000000

__degrees__

use constant H_RADIANS => <<'__radians__';
RADIANS(X):

     Returns the argument `X', converted from degrees to radians.
          mysql> select RADIANS(90);
                  -> 1.570796

__radians__

use constant H_TRUNCATE => <<'__truncate__';
TRUNCATE(X,D):

     Returns the number `X', truncated to `D' decimals.  If `D' is `0',
     the result will have no decimal point or fractional part.
          mysql> select TRUNCATE(1.223,1);
                  -> 1.2
          mysql> select TRUNCATE(1.999,1);
                  -> 1.9
          mysql> select TRUNCATE(1.999,0);
                  -> 1

__truncate__

use constant H_NUMERIC_FUNCTIONS => <<'__numeric_functions__';
Numeric Functions:

    abs
    sign
    mod
    floor
    ceiling
    round
    round
    exp
    log
    log10
    pow
    sqrt
    pi
    cos
    sin
    tan
    acos
    asin
    atan
    atan2
    cot
    rand
    least
    greatest
    degrees
    radians
    truncate

__numeric_functions__

use constant H_ASCII => <<'__ascii__';
ASCII(str):

     Returns the ASCII code value of the leftmost character of the
     string `str'. Returns `0' if `str' is the empty string.  Returns
     `NULL' if `str' is `NULL'.

          mysql> select ASCII('2');
                  -> 50
          mysql> select ASCII(2);
                  -> 50
          mysql> select ASCII('dx');
                  -> 100

     See also the `ORD()' function.

__ascii__

use constant H_ORD => <<'__ord__';
ORD(str):

     If the leftmost character of the string str is a multi-byte
     character, returns the code of multi-byte character by returning
     the ASCII code value of the character in the format of: `((first
     byte ASCII code)*256+(second byte ASCII code))[*256+third byte
     ASCII code...]'.  If the leftmost character is not a multi-byte
     character, returns the same value as the like `ASCII()' function
     does.

          mysql> select ORD('2');
                  -> 50

__ord__

use constant H_CONV => <<'__conv__';
CONV(N,from_base,to_base):

     Converts numbers between different number bases.  Returns a string
     representation of the number `N', converted from base `from_base'
     to base `to_base'.  Returns `NULL' if any argument is `NULL'.  The
     argument `N' is interpreted as an integer, but may be specified as
     an integer or a string.  The minimum base is `2' and the maximum
     base is `36'.  If `to_base' is a negative number, `N' is regarded
     as a signed number.  Otherwise, `N' is treated as unsigned.
     `CONV' works with 64-bit precision.

          mysql> select CONV("a",16,2);
                  -> '1010'
          mysql> select CONV("6E",18,8);
                  -> '172'
          mysql> select CONV(-17,10,-18);
                  -> '-H'
          mysql> select CONV(10+"10"+'10'+0xa,10,10);
                  -> '40'

__conv__

use constant H_BIN => <<'__bin__';
BIN(N):

     Returns a string representation of the binary value of `N', where
     `N' is a longlong (`BIGINT') number.  This is equivalent to
     `CONV(N,10,2)'.  Returns `NULL' if `N' is `NULL'.

          mysql> select BIN(12);
                  -> '1100'

__bin__

use constant H_OCT => <<'__oct__';
OCT(N):

     Returns a string representation of the octal value of `N', where
     `N' is a longlong number.  This is equivalent to `CONV(N,10,8)'.
     Returns `NULL' if `N' is `NULL'.

          mysql> select OCT(12);
                  -> '14'

__oct__

use constant H_HEX => <<'__hex__';
HEX(N):

     Returns a string representation of the hexadecimal value of `N',
     where `N' is a longlong (`BIGINT') number.  This is equivalent to
     `CONV(N,10,16)'.  Returns `NULL' if `N' is `NULL'.

          mysql> select HEX(255);
                  -> 'FF'

__hex__

use constant H_CHAR => <<'__char__';
CHAR(N,...):

     `CHAR()' interprets the arguments as integers and returns a string
     consisting of the characters given by the ASCII code values of
     those integers. `NULL' values are skipped.

          mysql> select CHAR(77,121,83,81,'76');
                  -> 'MySQL'
          mysql> select CHAR(77,77.3,'77.3');
                  -> 'MMM'

__char__

use constant H_CONCAT => <<'__concat__';
CONCAT(str1,str2,...):

     Returns the string that results from concatenating the arguments.
     Returns `NULL' if any argument is `NULL'.  May have more than 2
     arguments.  A numeric argument is converted to the equivalent
     string form.
          mysql> select CONCAT('My', 'S', 'QL');
                  -> 'MySQL'
          mysql> select CONCAT('My', NULL, 'QL');
                  -> NULL
          mysql> select CONCAT(14.3);
                  -> '14.3'

__concat__

use constant H_CONCAT_WS => <<'__concat_ws__';
CONCAT_WS(separator, str1, str2,...):

     `CONCAT_WS()' stands for CONCAT With Separator and is a special
     form of `CONCAT()'.  The irst argument is the separator for the
     rest of the arguments. The separator can be a string as well as
     the rest of the arguments. If the separator is `NULL', the result
     will be `NULL'.  The function will skip any `NULL's and empty
     strings, after the separator argument. The separator will be added
     between the strings to be concatenated.
          mysql> select CONCAT_WS(",","First name","Second name","Last Name");
                 -> 'First name,Second name,Last Name'
          mysql> select CONCAT_WS(",","First name",NULL,"Last Name");
                 -> 'First name,Last Name'

__concat_ws__

use constant H_LENGTH => <<'__length__';
LENGTH(str):

__length__

use constant H_OCTET_LENGTH => <<'__octet_length__';
OCTET_LENGTH(str):

__octet_length__

use constant H_CHAR_LENGTH => <<'__char_length__';
CHAR_LENGTH(str):

__char_length__

use constant H_CHARACTER_LENGTH => <<'__character_length__';
CHARACTER_LENGTH(str):

     Returns the length of the string `str'.
          mysql> select LENGTH('text');
                  -> 4
          mysql> select OCTET_LENGTH('text');
                  -> 4

     Note that for `CHAR_LENGTH()', multi-byte characters are only
     counted once.

__character_length__

use constant H_POSITION => <<'__position__';
POSITION(substr IN str):

     Returns the position of the first occurrence of substring `substr'
     in string `str'. Returns `0' if `substr' is not in `str'.
          mysql> select LOCATE('bar', 'foobarbar');
                  -> 4
          mysql> select LOCATE('xbar', 'foobar');
                  -> 0

     This function is multi-byte safe.

__position__

use constant H_LOCATE => <<'__locate__';
LOCATE(substr,str):
LOCATE(substr,str,pos):

     Returns the position of the first occurrence of substring `substr'
     in string `str', starting at position `pos'.  Returns `0' if
     `substr' is not in `str'.
          mysql> select LOCATE('bar', 'foobarbar',5);
                  -> 7

     This function is multi-byte safe.

__locate__

use constant H_INSTR => <<'__instr__';
INSTR(str,substr):

     Returns the position of the first occurrence of substring `substr'
     in string `str'. This is the same as the two-argument form of
     `LOCATE()', except that the arguments are swapped.

          mysql> select INSTR('foobarbar', 'bar');
                  -> 4
          mysql> select INSTR('xbar', 'foobar');
                  -> 0

     This function is multi-byte safe.

__instr__

use constant H_LPAD => <<'__lpad__';
LPAD(str,len,padstr):

     Returns the string `str', left-padded with the string `padstr'
     until `str' is `len' characters long.
          mysql> select LPAD('hi',4,'??');
                  -> '??hi'

__lpad__

use constant H_RPAD => <<'__rpad__';
RPAD(str,len,padstr):

     Returns the string `str', right-padded with the string `padstr'
     until `str' is `len' characters long.
          mysql> select RPAD('hi',5,'?');
                  -> 'hi???'

__rpad__

use constant H_LEFT => <<'__left__';
LEFT(str,len):

     Returns the leftmost `len' characters from the string `str'.
          mysql> select LEFT('foobarbar', 5);
                  -> 'fooba'

     This function is multi-byte safe.

__left__

use constant H_RIGHT => <<'__right__';
RIGHT(str,len):

     Returns the rightmost `len' characters from the string `str'.
          mysql> select RIGHT('foobarbar', 4);
                  -> 'rbar'

     This function is multi-byte safe.

__right__

use constant H_MID => <<'__mid__';
MID(str,pos,len):

     Returns a substring `len' characters long from string `str',
     starting at position `pos'.  The variant form that uses `FROM' is
     ANSI SQL92 syntax.
          mysql> select SUBSTRING('Quadratically',5,6);
                  -> 'ratica'

     This function is multi-byte safe.

__mid__

use constant H_SUBSTRING => <<'__substring__';
SUBSTRING(str,pos,len):
SUBSTRING(str FROM pos FOR len):
SUBSTRING(str,pos):
SUBSTRING(str FROM pos):

     Returns a substring from string `str' starting at position `pos'.
          mysql> select SUBSTRING('Quadratically',5);
                  -> 'ratically'
          mysql> select SUBSTRING('foobarbar' FROM 4);
                  -> 'barbar'

     This function is multi-byte safe.

__substring__

use constant H_SUBSTRING_INDEX => <<'__substring_index__';
SUBSTRING_INDEX(str,delim,count):

     Returns the substring from string `str' before `count' occurrences
     of the delimiter `delim'.  If `count' is positive, everything to
     the left of the final delimiter (counting from the left) is
     returned.  If `count' is negative, everything to the right of the
     final delimiter (counting from the right) is returned.
          mysql> select SUBSTRING_INDEX('www.mysql.com', '.', 2);
                  -> 'www.mysql'
          mysql> select SUBSTRING_INDEX('www.mysql.com', '.', -2);
                  -> 'mysql.com'

     This function is multi-byte safe.

__substring_index__

use constant H_LTRIM => <<'__ltrim__';
LTRIM(str):

     Returns the string `str' with leading space characters removed.
          mysql> select LTRIM('  barbar');
                  -> 'barbar'

__ltrim__

use constant H_RTRIM => <<'__rtrim__';
RTRIM(str):

     Returns the string `str' with trailing space characters removed.
          mysql> select RTRIM('barbar   ');
                  -> 'barbar'

     This function is multi-byte safe.

__rtrim__

use constant H_TRIM => <<'__trim__';
TRIM([[BOTH | LEADING | TRAILING] [remstr] FROM] str):

     Returns the string `str' with all `remstr' prefixes and/or suffixes
     removed. If none of the specifiers `BOTH', `LEADING' or `TRAILING'
     are given, `BOTH' is assumed. If `remstr' is not specified, spaces
     are removed.
          mysql> select TRIM('  bar   ');
                  -> 'bar'
          mysql> select TRIM(LEADING 'x' FROM 'xxxbarxxx');
                  -> 'barxxx'
          mysql> select TRIM(BOTH 'x' FROM 'xxxbarxxx');
                  -> 'bar'
          mysql> select TRIM(TRAILING 'xyz' FROM 'barxxyz');
                  -> 'barx'

     This function is multi-byte safe.

__trim__

use constant H_SOUNDEX => <<'__soundex__';
SOUNDEX(str):

     Returns a soundex string from `str'. Two strings that sound "about
     the same" should have identical soundex strings. A "standard"
     soundex string is 4 characters long, but the `SOUNDEX()' function
     returns an arbitrarily long string. You can use `SUBSTRING()' on
     the result to get a "standard" soundex string.  All
     non-alphanumeric characters are ignored in the given string. All
     international alpha characters outside the A-Z range are treated
     as vowels.

          mysql> select SOUNDEX('Hello');
                  -> 'H400'
          mysql> select SOUNDEX('Quadratically');
                  -> 'Q36324'

__soundex__

use constant H_SPACE => <<'__space__';
SPACE(N):

     Returns a string consisting of `N' space characters.
          mysql> select SPACE(6);
                  -> '      '

__space__

use constant H_FN_REPLACE => <<'__replace__';
REPLACE(str,from_str,to_str):

     Returns the string `str' with all all occurrences of the string
     `from_str' replaced by the string `to_str'.

          mysql> select REPLACE('www.mysql.com', 'w', 'Ww');
                  -> 'WwWwWw.mysql.com'

     This function is multi-byte safe.

__replace__

use constant H_REPEAT => <<'__repeat__';
REPEAT(str,count):

     Returns a string consisting of the string `str' repeated `count'
     times. If `count <= 0', returns an empty string. Returns `NULL' if
     `str' or `count' are `NULL'.

          mysql> select REPEAT('MySQL', 3);
                  -> 'MySQLMySQLMySQL'

__repeat__

use constant H_REVERSE => <<'__reverse__';
REVERSE(str):

     Returns the string `str' with the order of the characters reversed.
          mysql> select REVERSE('abc');
                  -> 'cba'

     This function is multi-byte safe.

__reverse__

use constant H_FN_INSERT => <<'__insert__';
INSERT(str,pos,len,newstr):

     Returns the string `str', with the substring beginning at position
     `pos' and `len' characters long replaced by the string `newstr'.

          mysql> select INSERT('Quadratic', 3, 4, 'What');
                  -> 'QuWhattic'

     This function is multi-byte safe.

__insert__

use constant H_ELT => <<'__elt__';
ELT(N,str1,str2,str3,...):

     Returns `str1' if `N' = `1', `str2' if `N' = `2', and so on.
     Returns `NULL' if `N' is less than `1' or greater than the number
     of arguments.  `ELT()' is the complement of `FIELD()'.

          mysql> select ELT(1, 'ej', 'Heja', 'hej', 'foo');
                  -> 'ej'
          mysql> select ELT(4, 'ej', 'Heja', 'hej', 'foo');
                  -> 'foo'

__elt__

use constant H_FIELD => <<'__field__';
FIELD(str,str1,str2,str3,...):

     Returns the index of `str' in the `str1', `str2', `str3', `...'
     list.  Returns `0' if `str' is not found.  `FIELD()' is the
     complement of `ELT()'.

          mysql> select FIELD('ej', 'Hej', 'ej', 'Heja', 'hej', 'foo');
                  -> 2
          mysql> select FIELD('fo', 'Hej', 'ej', 'Heja', 'hej', 'foo');
                  -> 0

__field__

use constant H_FIND_IN_SET => <<'__find_in_set__';
FIND_IN_SET(str,strlist):

     Returns a value `1' to `N' if the string `str' is in the list
     `strlist' consisting of `N' substrings. A string list is a string
     composed of substrings separated by `,' characters. If the first
     argument is a constant string and the second is a column of type
     `SET', the `FIND_IN_SET()' function is optimized to use bit
     arithmetic!  Returns `0' if `str' is not in `strlist' or if
     `strlist' is the empty string.  Returns `NULL' if either argument
     is `NULL'.  This function will not work properly if the first
     argument contains a `,'.

          mysql> SELECT FIND_IN_SET('b','a,b,c,d');
                  -> 2

__find_in_set__

use constant H_MAKE_SET => <<'__make_set__';
MAKE_SET(bits,str1,str2,...):

     Returns a set (a string containing substrings separated by `,'
     characters) consisting of the strings that have the corresponding
     bit in `bits' set.  `str1' corresponds to bit 0, `str2' to bit 1,
     etc.  `NULL' strings in `str1', `str2', `...' are not appended to
     the result.

          mysql> SELECT MAKE_SET(1,'a','b','c');
                  -> 'a'
          mysql> SELECT MAKE_SET(1 | 4,'hello','nice','world');
                  -> 'hello,world'
          mysql> SELECT MAKE_SET(0,'a','b','c');
                  -> ''

__make_set__

use constant H_EXPORT_SET => <<'__export_set__';
EXPORT_SET(bits,on,off,[separator,[number_of_bits]]):

     Returns a string where for every bit set in 'bit', you get a 'on'
     string and for every reset bit you get an 'off' string. Each
     string is separated with 'separator' (default ',') and only
     'number_of_bits' (default 64) of 'bits' is used.

          mysql> select EXPORT_SET(5,'Y','N',',',4)
                  -> Y,N,Y,N

__export_set__

use constant H_LCASE => <<'__lcase__';
LCASE(str):

__lcase__

use constant H_LOWER => <<'__lower__';
LOWER(str):

     Returns the string `str' with all characters changed to lowercase
     according to the current character set mapping (the default is
     ISO-8859-1 Latin1).

     This function is multi-byte safe.

          mysql> select LCASE('QUADRATICALLY');
                  -> 'quadratically'

__lower__

use constant H_UCASE => <<'__ucase__';
UCASE(str):

__ucase__

use constant H_UPPER => <<'__upper__';
UPPER(str):

     Returns the string `str' with all characters changed to uppercase
     according to the current character set mapping (the default is
     ISO-8859-1 Latin1).

          mysql> select UCASE('Hej');
                  -> 'HEJ'

     This function is multi-byte safe.

__upper__

use constant H_LOAD_FILE => <<'__load_file__';
LOAD_FILE(file_name):

     Reads the file and returns the file contents as a string.  The file
     must be on the server, you must specify the full pathname to the
     file, and you must have the *file* privilege.  The file must be
     readable by all and be smaller than `max_allowed_packet'.

     If the file doesn't exist or can't be read due to one of the above
     reasons, the function returns `NULL'.

          mysql> UPDATE table_name
                     SET blob_column=LOAD_FILE("/tmp/picture")
                     WHERE id=1;

If you are not using *MySQL 3.23*, you have to do the reading of the
file inside your application and create an `INSERT' statement to update
the database with the file information. One way to do this, if you are
using the *MySQL*++ library, can be found at
`http://www.mysql.com/documentation/mysql++/mysql++-examples.html'.

*MySQL* automatically converts numbers to strings as necessary, and
vice versa:

     mysql> SELECT 1+"1";
             -> 2
     mysql> SELECT CONCAT(2,' test');
             -> '2 test'

If you want to convert a number to a string explicitly, pass it as the
argument to `CONCAT()'.

If a string function is given a binary string as an argument, the
resulting string is also a binary string.  A number converted to a
string is treated as a binary string.  This only affects comparisons.

__load_file__

use constant H_STRING_FUNCTIONS => <<'__string_functions__';
String Functions:

String-valued functions return `NULL' if the length of the result would
be greater than the `max_allowed_packet' server parameter.

For functions that operate on string positions, the first position is
numbered 1.

    ascii
    ord
    conv
    bin
    oct
    hex
    char
    concat
    concat_ws
    length
    octet_length
    char_length
    character_length
    locate
    position
    instr
    lpad
    rpad
    left
    right
    substring
    substring
    mid
    substring
    substring_index
    ltrim
    rtrim
    trim
    soundex
    space
    replace [see fn_replace]
    repeat
    reverse
    insert  [see fn_insert]
    elt
    field
    find_in_set
    make_set
    export_set
    lcase
    lower
    ucase
    upper
    load_file

__string_functions__

use constant H_DAYOFWEEK => <<'__dayofweek__';
DAYOFWEEK(date):

     Returns the weekday index for `date' (`1' = Sunday, `2' = Monday,
     ... `7' = Saturday).  These index values correspond to the ODBC
     standard.
          mysql> select DAYOFWEEK('1998-02-03');
                  -> 3

__dayofweek__

use constant H_WEEKDAY => <<'__weekday__';
WEEKDAY(date):

     Returns the weekday index for `date' (`0' = Monday, `1' = Tuesday,
     ... `6' = Sunday).
          mysql> select WEEKDAY('1997-10-04 22:23:00');
                  -> 5
          mysql> select WEEKDAY('1997-11-05');
                  -> 2

__weekday__

use constant H_DAYOFMONTH => <<'__dayofmonth__';
DAYOFMONTH(date):

     Returns the day of the month for `date', in the range `1' to `31'.
          mysql> select DAYOFMONTH('1998-02-03');
                  -> 3

__dayofmonth__

use constant H_DAYOFYEAR => <<'__dayofyear__';
DAYOFYEAR(date):

     Returns the day of the year for `date', in the range `1' to `366'.
          mysql> select DAYOFYEAR('1998-02-03');
                  -> 34

__dayofyear__

use constant H_MONTH => <<'__month__';
MONTH(date):

     Returns the month for `date', in the range `1' to `12'.
          mysql> select MONTH('1998-02-03');
                  -> 2

__month__

use constant H_DAYNAME => <<'__dayname__';
DAYNAME(date):

     Returns the name of the weekday for `date'.
          mysql> select DAYNAME("1998-02-05");
                  -> 'Thursday'

__dayname__

use constant H_MONTHNAME => <<'__monthname__';
MONTHNAME(date):

     Returns the name of the month for `date'.
          mysql> select MONTHNAME("1998-02-05");
                  -> 'February'

__monthname__

use constant H_QUARTER => <<'__quarter__';
QUARTER(date):

     Returns the quarter of the year for `date', in the range `1' to
     `4'.
          mysql> select QUARTER('98-04-01');
                  -> 2

__quarter__

use constant H_WEEK => <<'__week__';
WEEK(date):
WEEK(date,first):

     With a single argument, returns the week for `date', in the range
     `0' to `53' (yes, there may be the beginnings of a week 53), for
     locations where Sunday is the first day of the week.  The
     two-argument form of `WEEK()' allows you to specify whether the
     week starts on Sunday or Monday.  The week starts on Sunday if the
     second argument is `0', on Monday if the second argument is `1'.
          mysql> select WEEK('1998-02-20');
                  -> 7
          mysql> select WEEK('1998-02-20',0);
                  -> 7
          mysql> select WEEK('1998-02-20',1);
                  -> 8
          mysql> select WEEK('1998-12-31',1);
                  -> 53

__week__

use constant H_YEAR => <<'__year__';
YEAR(date):

     Returns the year for `date', in the range `1000' to `9999'.
          mysql> select YEAR('98-02-03');
                  -> 1998

__year__

use constant H_YEARWEEK => <<'__yearweek__';
YEARWEEK(date):
YEARWEEK(date,first):

     Returns year and week for a date.  The second arguments works
     exactly like the second argument to `WEEK()'.  Note that the year
     may be different from the year in the date argument for the first
     and the last week of the year!
          mysql> select YEARWEEK('1987-01-01');
                  -> 198653

__yearweek__

use constant H_HOUR => <<'__hour__';
HOUR(time):

     Returns the hour for `time', in the range `0' to `23'.
          mysql> select HOUR('10:05:03');
                  -> 10

__hour__

use constant H_MINUTE => <<'__minute__';
MINUTE(time):

     Returns the minute for `time', in the range `0' to `59'.
          mysql> select MINUTE('98-02-03 10:05:03');
                  -> 5

__minute__

use constant H_SECOND => <<'__second__';
SECOND(time):

     Returns the second for `time', in the range `0' to `59'.
          mysql> select SECOND('10:05:03');
                  -> 3

__second__

use constant H_PERIOD_ADD => <<'__period_add__';
PERIOD_ADD(P,N):

     Adds `N' months to period `P' (in the format `YYMM' or `YYYYMM').
     Returns a value in the format `YYYYMM'.

     Note that the period argument `P' is _not_ a date value.

          mysql> select PERIOD_ADD(9801,2);
                  -> 199803

__period_add__

use constant H_PERIOD_DIFF => <<'__period_diff__';
PERIOD_DIFF(P1,P2):

     Returns the number of months between periods `P1' and `P2'.  `P1'
     and `P2' should be in the format `YYMM' or `YYYYMM'.

     Note that the period arguments `P1' and `P2' are _not_ date values.

          mysql> select PERIOD_DIFF(9802,199703);
                  -> 11

__period_diff__

use constant H_DATE_ADD => <<'__date_add__';
DATE_ADD(date,INTERVAL expr type):
DATE_SUB(date,INTERVAL expr type):
ADDDATE(date,INTERVAL expr type):
SUBDATE(date,INTERVAL expr type):

     These functions perform date arithmetic.  They are new for *MySQL*
     3.22.  `ADDDATE()' and `SUBDATE()' are synonyms for `DATE_ADD()'
     and `DATE_SUB()'.

     In *MySQL* 3.23, you can use `+' and `-' instead of `DATE_ADD()'
     and `DATE_SUB()'. (See example)

     `date' is a `DATETIME' or `DATE' value specifying the starting
     date.  `expr' is an expression specifying the interval value to be
     added or substracted from the starting date.  `expr' is a string;
     it may start with a `-' for negative intervals.  `type' is a
     keyword indicating how the expression should be interpreted.

     The `EXTRACT(type FROM date)' function returns the 'type' interval
     from the date.

     The following table shows how the `type' and `expr' arguments are
     related:

     `type'       *Meaning*            *Expected* `expr' *format*
     *value*                           
     `SECOND'     Seconds              `SECONDS'
     `MINUTE'     Minutes              `MINUTES'
     `HOUR'       Hours                `HOURS'
     `DAY'        Days                 `DAYS'
     `MONTH'      Months               `MONTHS'
     `YEAR'       Years                `YEARS'
     `MINUTE_SECOND'Minutes and seconds  `"MINUTES:SECONDS"'
     `HOUR_MINUTE'Hours and minutes    `"HOURS:MINUTES"'
     `DAY_HOUR'   Days and hours       `"DAYS HOURS"'
     `YEAR_MONTH' Years and months     `"YEARS-MONTHS"'
     `HOUR_SECOND'Hours, minutes,      `"HOURS:MINUTES:SECONDS"'
     `DAY_MINUTE' Days, hours,         `"DAYS HOURS:MINUTES"'
                  minutes              
     `DAY_SECOND' Days, hours,         `"DAYS
                  minutes, seconds     HOURS:MINUTES:SECONDS"'

     *MySQL* allows any punctuation delimiter in the `expr' format.
     The ones shown in the table are the suggested delimiters.  If the
     `date' argument is a `DATE' value and your calculations involve
     only `YEAR', `MONTH' and `DAY' parts (that is, no time parts), the
     result is a `DATE' value.  Otherwise the result is a `DATETIME'
     value.

          mysql> SELECT "1997-12-31 23:59:59" + INTERVAL 1 SECOND;
                  -> 1998-01-01 00:00:00
          mysql> SELECT INTERVAL 1 DAY + "1997-12-31";
                  -> 1998-01-01
          mysql> SELECT "1998-01-01" - INTERVAL 1 SECOND;
                 -> 1997-12-31 23:59:59
          mysql> SELECT DATE_ADD("1997-12-31 23:59:59",
                                 INTERVAL 1 SECOND);
                  -> 1998-01-01 00:00:00
          mysql> SELECT DATE_ADD("1997-12-31 23:59:59",
                                 INTERVAL 1 DAY);
                  -> 1998-01-01 23:59:59
          mysql> SELECT DATE_ADD("1997-12-31 23:59:59",
                                 INTERVAL "1:1" MINUTE_SECOND);
                  -> 1998-01-01 00:01:00
          mysql> SELECT DATE_SUB("1998-01-01 00:00:00",
                                 INTERVAL "1 1:1:1" DAY_SECOND);
                  -> 1997-12-30 22:58:59
          mysql> SELECT DATE_ADD("1998-01-01 00:00:00",
                                 INTERVAL "-1 10" DAY_HOUR);
                  -> 1997-12-30 14:00:00
          mysql> SELECT DATE_SUB("1998-01-02", INTERVAL 31 DAY);
                  -> 1997-12-02
          mysql> SELECT EXTRACT(YEAR FROM "1999-07-02");
                 -> 1999
          mysql> SELECT EXTRACT(YEAR_MONTH FROM "1999-07-02 01:02:03");
                 -> 199907
          mysql> SELECT EXTRACT(DAY_MINUTE FROM "1999-07-02 01:02:03");
                 -> 20102

     If you specify an interval value that is too short (does not
     include all the interval parts that would be expected from the
     `type' keyword), *MySQL* assumes you have left out the leftmost
     parts of the interval value.  For example, if you specify a `type'
     of `DAY_SECOND', the value of `expr' is expected to have days,
     hours, minutes and seconds parts.  If you specify a value like
     `"1:10"', *MySQL* assumes that the days and hours parts are
     missing and the value represents minutes and seconds.  In other
     words, `"1:10" DAY_SECOND' is interpreted in such a way that it is
     equivalent to `"1:10" MINUTE_SECOND'.  This is analogous to the
     way that *MySQL* interprets `TIME' values as representing elapsed
     time rather than as time of day.

     If you use really incorrect dates, the result is `NULL'. If you add
     `MONTH', `YEAR_MONTH' or `YEAR' and the resulting date has a day
     that is larger than the maximum day for the new month, the day is
     adjusted to the maximum days in the new month.

          mysql> select DATE_ADD('1998-01-30', Interval 1 month);
                  -> 1998-02-28

     Note from the preceding example that the word `INTERVAL' and the
     `type' keyword are not case sensitive.

__date_add__

use constant H_DATE_SUB => H_DATE_ADD;
use constant H_ADDDATE  => H_DATE_ADD;
use constant H_SUBDATE  => H_DATE_ADD;

use constant H_TO_DAYS => <<'__to_days__';
TO_DAYS(date):

     Given a date `date', returns a daynumber (the number of days since
     year 0).

          mysql> select TO_DAYS(950501);
                  -> 728779
          mysql> select TO_DAYS('1997-10-07');
                  -> 729669

     `TO_DAYS()' is not intended for use with values that precede the
     advent of the Gregorian calendar (1582), because it doesn't take
     into account the days that were lost when the calender was changed.

__to_days__

use constant H_FROM_DAYS => <<'__from_days__';
FROM_DAYS(N):

     Given a daynumber `N', returns a `DATE' value.

          mysql> select FROM_DAYS(729669);
                  -> '1997-10-07'

     `FROM_DAYS()' is not intended for use with values that precede the
     advent of the Gregorian calendar (1582), because it doesn't take
     into account the days that were lost when the calender was changed.

__from_days__

use constant H_DATE_FORMAT => <<'__date_format__';
DATE_FORMAT(date,format):

     Formats the `date' value according to the `format' string. The
     following specifiers may be used in the `format' string:
     `%M'    Month name (`January'..`December')
     `%W'    Weekday name (`Sunday'..`Saturday')
     `%D'    Day of the month with english suffix (`1st', `2nd', `3rd',
             etc.)
     `%Y'    Year, numeric, 4 digits
     `%y'    Year, numeric, 2 digits
     `%X'    Year for the week where Sunday is the first day of the
             week, numeric, 4 digits, used with '%V'
     `%x'    Year for the week, where Monday is the first day of the
             week, numeric, 4 digits, used with '%v'
     `%a'    Abbreviated weekday name (`Sun'..`Sat')
     `%d'    Day of the month, numeric (`00'..`31')
     `%e'    Day of the month, numeric (`0'..`31')
     `%m'    Month, numeric (`01'..`12')
     `%c'    Month, numeric (`1'..`12')
     `%b'    Abbreviated month name (`Jan'..`Dec')
     `%j'    Day of year (`001'..`366')
     `%H'    Hour (`00'..`23')
     `%k'    Hour (`0'..`23')
     `%h'    Hour (`01'..`12')
     `%I'    Hour (`01'..`12')
     `%l'    Hour (`1'..`12')
     `%i'    Minutes, numeric (`00'..`59')
     `%r'    Time, 12-hour (`hh:mm:ss [AP]M')
     `%T'    Time, 24-hour (`hh:mm:ss')
     `%S'    Seconds (`00'..`59')
     `%s'    Seconds (`00'..`59')
     `%p'    `AM' or `PM'
     `%w'    Day of the week (`0'=Sunday..`6'=Saturday)
     `%U'    Week (`0'..`53'), where Sunday is the first day of the week
     `%u'    Week (`0'..`53'), where Monday is the first day of the week
     `%V'    Week (`1'..`53'), where Sunday is the first day of the
             week. Used with '%X'
     `%v'    Week (`1'..`53'), where Monday is the first day of the
             week. Used with '%x'
     `%%'    A literal `%'.

     All other characters are just copied to the result without
     interpretation.

          mysql> select DATE_FORMAT('1997-10-04 22:23:00', '%W %M %Y');
                  -> 'Saturday October 1997'
          mysql> select DATE_FORMAT('1997-10-04 22:23:00', '%H:%i:%s');
                  -> '22:23:00'
          mysql> select DATE_FORMAT('1997-10-04 22:23:00',
                                    '%D %y %a %d %m %b %j');
                  -> '4th 97 Sat 04 10 Oct 277'
          mysql> select DATE_FORMAT('1997-10-04 22:23:00',
                                    '%H %k %I %r %T %S %w');
                  -> '22 22 10 10:23:00 PM 22:23:00 00 6'
          mysql> select DATE_FORMAT('1999-01-01', '%X %V');
                  -> '1998 52'

     As of *MySQL* 3.23, the `%' character is required before format
     specifier characters.  In earlier versions of *MySQL*, `%' was
     optional.

__date_format__

use constant H_TIME_FORMAT => <<'__time_format__';
TIME_FORMAT(time,format):

     This is used like the `DATE_FORMAT()' function above, but the
     `format' string may contain only those format specifiers that
     handle hours, minutes and seconds.  Other specifiers produce a
     `NULL' value or `0'.

__time_format__

use constant H_CURDATE => <<'__curdate__';
CURDATE():
CURRENT_DATE:

     Returns today's date as a value in `'YYYY-MM-DD'' or `YYYYMMDD'
     format, depending on whether the function is used in a string or
     numeric context.

          mysql> select CURDATE();
                  -> '1997-12-15'
          mysql> select CURDATE() + 0;
                  -> 19971215

__curdate__

use constant H_CURTIME => <<'__curtime__';
CURTIME():
CURRENT_TIME:

     Returns the current time as a value in `'HH:MM:SS'' or `HHMMSS'
     format, depending on whether the function is used in a string or
     numeric context.

          mysql> select CURTIME();
                  -> '23:50:26'
          mysql> select CURTIME() + 0;
                  -> 235026

__curtime__

use constant H_NOW => <<'__now__';
NOW():
SYSDATE():
CURRENT_TIMESTAMP:

     Returns the current date and time as a value in `'YYYY-MM-DD
     HH:MM:SS'' or `YYYYMMDDHHMMSS' format, depending on whether the
     function is used in a string or numeric context.

          mysql> select NOW();
                  -> '1997-12-15 23:50:26'
          mysql> select NOW() + 0;
                  -> 19971215235026

__now__

use constant H_SYSDATE => H_NOW;

use constant H_UNIX_TIMESTAMP => <<'__unix_timestamp__';
UNIX_TIMESTAMP():
UNIX_TIMESTAMP(date):

     If called with no argument, returns a Unix timestamp (seconds since
     `'1970-01-01 00:00:00'' GMT). If `UNIX_TIMESTAMP()' is called with
     a `date' argument, it returns the value of the argument as seconds
     since `'1970-01-01 00:00:00'' GMT.  `date' may be a `DATE' string,
     a `DATETIME' string, a `TIMESTAMP', or a number in the format
     `YYMMDD' or `YYYYMMDD' in local time.

          mysql> select UNIX_TIMESTAMP();
                  -> 882226357
          mysql> select UNIX_TIMESTAMP('1997-10-04 22:23:00');
                  -> 875996580

     When `UNIX_TIMESTAMP' is used on a `TIMESTAMP' column, the function
     will receive the value directly, with no implicit
     "string-to-unix-timestamp" conversion.  If you give
     `UNIX_TIMESTAMP()' a wrong or out-of-range date, it will return 0.

__unix_timestamp__

use constant H_FROM_UNIXTIME => <<'__from_unixtime__';
FROM_UNIXTIME(unix_timestamp):

     Returns a representation of the `unix_timestamp' argument as a
     value in `'YYYY-MM-DD HH:MM:SS'' or `YYYYMMDDHHMMSS' format,
     depending on whether the function is used in a string or numeric
     context.

          mysql> select FROM_UNIXTIME(875996580);
                  -> '1997-10-04 22:23:00'
          mysql> select FROM_UNIXTIME(875996580) + 0;
                  -> 19971004222300

FROM_UNIXTIME(unix_timestamp,format):

     Returns a string representation of the Unix timestamp, formatted
     according to the `format' string. `format' may contain the same
     specifiers as those listed in the entry for the `DATE_FORMAT()'
     function.

          mysql> select FROM_UNIXTIME(UNIX_TIMESTAMP(),
                                      '%Y %D %M %h:%i:%s %x');
                  -> '1997 23rd December 03:43:30 x'

__from_unixtime__

use constant H_SEC_TO_TIME => <<'__sec_to_time__';
SEC_TO_TIME(seconds):

     Returns the `seconds' argument, converted to hours, minutes and
     seconds, as a value in `'HH:MM:SS'' or `HHMMSS' format, depending
     on whether the function is used in a string or numeric context.

          mysql> select SEC_TO_TIME(2378);
                  -> '00:39:38'
          mysql> select SEC_TO_TIME(2378) + 0;
                  -> 3938

__sec_to_time__

use constant H_TIME_TO_SEC => <<'__time_to_sec__';
TIME_TO_SEC(time):

     Returns the `time' argument, converted to seconds.
          mysql> select TIME_TO_SEC('22:23:00');
                  -> 80580
          mysql> select TIME_TO_SEC('00:39:38');
                  -> 2378

__time_to_sec__

use constant H_DATE_TIME_FUNCTIONS => <<'__date_time_functions__';
Date and Time Functions:

    dayofweek
    weekday
    dayofmonth
    dayofyear
    month
    dayname
    monthname
    quarter
    week
    year
    yearweek
    hour
    minute
    second
    period_add
    period_diff
    date_add
    to_days
    from_days
    date_format
    time_format
    curdate
    curtime
    now
    unix_timestamp
    from_unixtime
    sec_to_time
    time_to_sec

__date_time_functions__

use constant H_DATABASE => <<'__database__';
DATABASE():

     Returns the current database name.
          mysql> select DATABASE();
                  -> 'test'

     If there is no current database, `DATABASE()' returns the empty
     string.

__database__

use constant H_USER => <<'__user__';
USER():
SYSTEM_USER():
SESSION_USER():

     Returns the current *MySQL* user name.
          mysql> select USER();
                  -> 'davida@localhost'

     In *MySQL* 3.22.11 or later, this includes the client hostname as
     well as the user name.  You can extract just the user name part
     like this (which works whether or not the value includes a
     hostname part):

          mysql> select substring_index(USER(),"@",1);
                  -> 'davida'

__user__

use constant H_SYSTEM_USER  => H_USER;
use constant H_SESSION_USER => H_USER;

use constant H_PASSWORD => <<'__password__';
PASSWORD(str):

     Calculates a password string from the plaintext password `str'.
     This is the function that is used for encrypting *MySQL* passwords
     for storage in the `Password' column of the `user' grant table.

          mysql> select PASSWORD('badpwd');
                  -> '7f84554057dd964b'

     `PASSWORD()' encryption is non-reversible.

     `PASSWORD()' does not perform password encryption in the same way
     that Unix passwords are encrypted.  You should not assume that if
     your Unix password and your *MySQL* password are the same,
     `PASSWORD()' will result in the same encrypted value as is stored
     in the Unix password file.  See `ENCRYPT()'.

__password__

use constant H_ENCRYPT => <<'__encrypt__';
ENCRYPT(str[,salt]):

     Encrypt `str' using the Unix `crypt()' system call. The `salt'
     argument should be a string with two characters.  (As of *MySQL*
     3.22.16, `salt' may be longer than two characters.)

          mysql> select ENCRYPT("hello");
                  -> 'VxuFAJXVARROc'

     If `crypt()' is not available on your system, `ENCRYPT()' always
     returns `NULL'.

     `ENCRYPT()' ignores all but the first 8 characters of `str', at
     least on some systems.  This will be determined by the behavior of
     the underlying `crypt()' system call.

__encrypt__

use constant H_ENCODE => <<'__encode__';
ENCODE(str,pass_str):

     Encrypt `str' using `pass_str' as the password.  To decrypt the
     result, use `DECODE()'.

     The results is a binary string of the same length as `string'.  If
     you want to save it in a column, use a `BLOB' column type.

__encode__

use constant H_DECODE => <<'__decode__';
DECODE(crypt_str,pass_str):

     Descrypts the encrypted string `crypt_str' using `pass_str' as the
     password.  `crypt_str' should be a string returned from `ENCODE()'.

__decode__

use constant H_MD5 => <<'__md5__';
MD5(string):

     Calculates a MD5 checksum for the string. Value is returned as a
     32 long hex number that may, for example, be used as a hash key.

          mysql> select MD5("testing")
                  -> 'ae2b1fca515949e5d54fb22b8ed95575'

     This is an "RSA Data Security, Inc. MD5 Message-Digest Algorithm".

__md5__

use constant H_LAST_INSERT_ID => <<'__last_insert_id__';
LAST_INSERT_ID([expr]):

     Returns the last automatically generated value that was inserted
     into an `AUTO_INCREMENT' column.  *Note `mysql_insert_id()':
     mysql_insert_id.

          mysql> select LAST_INSERT_ID();
                  -> 195

     The last ID that was generated is maintained in the server on a
     per-connection basis.  It will not be changed by another client.
     It will not even be changed if you update another `AUTO_INCREMENT'
     column with a non-magic value (that is, a value that is not `NULL'
     and not `0').

     If `expr' is given as an argument to `LAST_INSERT_ID()' in an
     `UPDATE' clause, then the value of the argument is returned as a
     `LAST_INSERT_ID()' value.  This can be used to simulate sequences:

     First create the table:

          mysql> create table sequence (id int not null);
          mysql> insert into sequence values (0);

     Then the table can be used to generate sequence numbers like this:

          mysql> update sequence set id=LAST_INSERT_ID(id+1);

     You can generate sequences without calling `LAST_INSERT_ID()', but
     the utility of using the function this way is that the ID value is
     maintained in the server as the last automatically generated
     value.  You can retrieve the new ID as you would read any normal
     `AUTO_INCREMENT' value in *MySQL*.  For example,
     `LAST_INSERT_ID()' (without an argument) will return the new ID.
     The C API function `mysql_insert_id()' can also be used to get the
     value.

__last_insert_id__

use constant H_FORMAT => <<'__format__';
FORMAT(X,D):

     Formats the number `X' to a format like `'#,###,###.##'', rounded
     to `D' decimals.  If `D' is `0', the result will have no decimal
     point or fractional part.

          mysql> select FORMAT(12332.123456, 4);
                  -> '12,332.1235'
          mysql> select FORMAT(12332.1,4);
                  -> '12,332.1000'
          mysql> select FORMAT(12332.2,0);
                  -> '12,332'

__format__

use constant H_VERSION => <<'__version__';
VERSION():

     Returns a string indicating the *MySQL* server version.
          mysql> select VERSION();
                  -> '3.23.13-log'

     Note that if your version ends with `-log' this means that logging
     is enabled.

__version__

use constant H_CONNECTION_ID => <<'__connection_id__';
CONNECTION_ID():

     Returns the connection id (`thread_id') for the connection.  Every
     connection has its own unique id.
          mysql> select CONNECTION_ID();
                  -> 1

__connection_id__

use constant H_GET_LOCK => <<'__get_lock__';
GET_LOCK(str,timeout):

     Tries to obtain a lock with a name given by the string `str', with
     a timeout of `timeout' seconds.  Returns `1' if the lock was
     obtained successfully, `0' if the attempt timed out, or `NULL' if
     an error occurred (such as running out of memory or the thread was
     killed with `mysqladmin kill').  A lock is released when you
     execute `RELEASE_LOCK()', execute a new `GET_LOCK()' or the thread
     terminates.  This function can be used to implement application
     locks or to simulate record locks.  It blocks requests by other
     clients for locks with the same name; clients that agree on a
     given lock string name can use the string to perform cooperative
     advisory locking.

          mysql> select GET_LOCK("lock1",10);
                  -> 1
          mysql> select GET_LOCK("lock2",10);
                  -> 1
          mysql> select RELEASE_LOCK("lock2");
                  -> 1
          mysql> select RELEASE_LOCK("lock1");
                  -> NULL

     Note that the second `RELEASE_LOCK()' call returns `NULL' because
     the lock `"lock1"' was automatically released by the second
     `GET_LOCK()' call.

__get_lock__

use constant H_RELEASE_LOCK => <<'__release_lock__';
RELEASE_LOCK(str):

     Releases the lock named by the string `str' that was obtained with
     `GET_LOCK()'. Returns `1' if the lock was released, `0' if the
     lock wasn't locked by this thread (in which case the lock is not
     released) and `NULL' if the named lock didn't exist.  The lock
     will not exist if it was never obtained by a call to `GET_LOCK()'
     or if it already has been released.

__release_lock__

use constant H_BENCHMARK => <<'__benchmark__';
BENCHMARK(count,expr):

     The `BENCHMARK()' function executes the expression `expr'
     repeatedly `count' times.  It may be used to time how fast *MySQL*
     processes the expression.  The result value is always `0'.  The
     intended use is in the `mysql' client, which reports query
     execution times.

          mysql> select BENCHMARK(1000000,encode("hello","goodbye"));
          +----------------------------------------------+
          | BENCHMARK(1000000,encode("hello","goodbye")) |
          +----------------------------------------------+
          |                                            0 |
          +----------------------------------------------+
          1 row in set (4.74 sec)

     The time reported is elapsed time on the client end, not CPU time
     on the server end.  It may be advisable to execute `BENCHMARK()'
     several times, and interpret the result with regard to how heavily
     loaded the server machine is.

__benchmark__

use constant H_INET_NTOA => <<'__inet_ntoa__';
INET_NTOA(expr):

     Returns the network address (4 or 8 byte) for the numeric
     expression.

          mysql> select INET_NTOA(3520061480);
                 ->  "209.207.224.40"

__inet_ntoa__

use constant H_INET_ATON => <<'__inet_aton__';
INET_ATON(expr):

     Returns an integer that represents the numeric value for a network
     address Addresses may be 4 or 8 byte addresses.

          mysql> select INET_ATON("209.207.224.40");
                 ->  3520061480

__inet_aton__

use constant H_MISC_FUNCTIONS => <<'__misc_functions__';
Miscellaneous Functions:

    database
    user
    password
    encrypt
    encode
    decode
    md5
    last_insert_id
    format
    version
    connection_id
    get_lock
    release_lock
    benchmark
    inet_ntoa
    inet_aton

__misc_functions__

use constant H_COUNT => <<'__count__';
COUNT(expr):

     Returns a count of the number of non-`NULL' values in the rows
     retrieved by a `SELECT' statement.

          mysql> select student.student_name,COUNT(*)
                     from student,course
                     where student.student_id=course.student_id
                     GROUP BY student_name;

     `COUNT(*)' is somewhat different in that it returns a count of the
     number of rows retrieved, whether or not they contain `NULL'
     values.

     `COUNT(*)' is optimized to return very quickly if the `SELECT'
     retrieves from one table, no other columns are retrieved and there
     is no `WHERE' clause.  For example:

          mysql> select COUNT(*) from student;

COUNT(DISTINCT expr,[expr...]):

     Returns a count of the number of different values.

          mysql> select COUNT(DISTINCT results) from student;

     In *MySQL* you can get the number of distinct expressions
     combinations by giving a list of expressions.  In ANSI SQL you
     would have to do a concatenation of all expressions inside
     `CODE(DISTINCT ..)'.

__count__

use constant H_AVG => <<'__avg__';
AVG(expr):

     Returns the average value of `expr'.

          mysql> select student_name, AVG(test_score)
                     from student
                     GROUP BY student_name;

__avg__


use constant H_MIN => <<'__min__';
MIN(expr):
MAX(expr):

     Returns the minimum or maximum value of `expr'.  `MIN()' and
     `MAX()' may take a string argument; in such cases they return the
     minimum or maximum string value.

          mysql> select student_name, MIN(test_score), MAX(test_score)
                     from student
                     GROUP BY student_name;

__min__

use constant H_MAX => H_MIN;

use constant H_SUM => <<'__sum__';
SUM(expr):

     Returns the sum of `expr'.  Note that if the return set has no
     rows, it returns NULL!

__sum__

use constant H_STD => <<'__std__';
STD(expr):
STDDEV(expr):

     Returns the standard deviation of `expr'. This is an extension to
     ANSI SQL.  The `STDDEV()' form of this function is provided for
     Oracle compatability.

__std__

use constant H_STDDEV => H_STD;

use constant H_BIT_OR => <<'__bit_or__';
BIT_OR(expr):

     Returns the bitwise `OR' of all bits in `expr'. The calculation is
     performed with 64-bit (`BIGINT') precision.

__bit_or__

use constant H_BIT_AND => <<'__bit_and__';
BIT_AND(expr):

     Returns the bitwise `AND' of all bits in `expr'. The calculation is
     performed with 64-bit (`BIGINT' precision.

__bit_and__

use constant H_GROUP_BY_FUNCTIONS => <<'__group_by_functions__';
Group By Functions:

*MySQL* has extended the use of `GROUP BY'. You can use columns or
calculations in the `SELECT' expressions which don't appear in the
`GROUP BY' part. This stands for _any possible value for this group_.
You can use this to get better performance by avoiding sorting and
grouping on unnecessary items.  For example, you don't need to group on
`customer.name' in the following query:

     mysql> select order.custid,customer.name,max(payments)
            from order,customer
            where order.custid = customer.custid
            GROUP BY order.custid;

In ANSI SQL, you would have to add `customer.name' to the `GROUP BY'
clause.  In *MySQL*, the name is redundant if you don't run in ANSI
mode.

Don't use this feature if the columns you omit from the `GROUP BY' part
aren't unique in the group!

In some cases, you can use `MIN()' and `MAX()' to obtain a specific
column value even if it isn't unique. The following gives the value of
`column' from the row containing the smallest value in the `sort'
column:

     substr(MIN(concat(sort,space(6-length(sort)),column),7,length(column)))

Note that if you are using *MySQL* 3.22 (or earlier) or if you are
trying to follow ANSI SQL, you can't use expressions in `GROUP BY' or
`ORDER BY' clauses.  You can work around this limitation by using an
alias for the expression:

     mysql> SELECT id,FLOOR(value/100) AS val FROM tbl_name
                GROUP BY id,val ORDER BY val;

In `MySQL' 3.23 you can do:

     mysql> SELECT id,FLOOR(value/100) FROM tbl_name ORDER BY RAND();

See:
    count
    avg
    min
    sum
    std
    bit_or
    bit_and

__group_by_functions__

use constant H_DATA_TYPES => <<'__data_types__';
Column types
============

*MySQL* supports a number of column types, which may be grouped into
three categories: numeric types, date and time types, and string
(character) types.  This section first gives an overview of the types
available and summarizes the storage requirements for each column type,
then provides a more detailed description of the properties of the
types in each category.  The overview is intentionally brief.  The more
detailed descriptions should be consulted for additional information
about particular column types, such as the allowable formats in which
you can specify values.

The column types supported by *MySQL* are listed below.  The following
code letters are used in the descriptions:

`M'
     Indicates the maximum display size.  The maximum legal display
     size is 255.

`D'
     Applies to floating-point types and indicates the number of digits
     following the decimal point.  The maximum possible value is 30, but
     should be no greater than `M'-2.

Square brackets (`[' and `]') indicate parts of type specifiers that
are optional.

Note that if you specify `ZEROFILL' for a column, *MySQL* will
automatically add the `UNSIGNED' attribute to the column.

`TINYINT[(M)] [UNSIGNED] [ZEROFILL]'
     A very small integer. The signed range is `-128' to `127'. The
     unsigned range is `0' to `255'.

`SMALLINT[(M)] [UNSIGNED] [ZEROFILL]'
     A small integer. The signed range is `-32768' to `32767'. The
     unsigned range is `0' to `65535'.

`MEDIUMINT[(M)] [UNSIGNED] [ZEROFILL]'
     A medium-size integer. The signed range is `-8388608' to
     `8388607'. The unsigned range is `0' to `16777215'.

`INT[(M)] [UNSIGNED] [ZEROFILL]'
     A normal-size integer. The signed range is `-2147483648' to
     `2147483647'.  The unsigned range is `0' to `4294967295'.

`INTEGER[(M)] [UNSIGNED] [ZEROFILL]'
     This is a synonym for `INT'.

`BIGINT[(M)] [UNSIGNED] [ZEROFILL]'
     A large integer. The signed range is `-9223372036854775808' to
     `9223372036854775807'. The unsigned range is `0' to
     `18446744073709551615'.  Note that all arithmetic is done using
     signed `BIGINT' or `DOUBLE' values, so you shouldn't use unsigned
     big integers larger than `9223372036854775807' (63 bits) except
     with bit functions!  Note that `-', `+' and `*' will use `BIGINT'
     arithmetic when both arguments are `INTEGER' values!  This means
     that if you multiply two big integers (or results from functions
     that return integers) you may get unexpected results if the result
     is larger than `9223372036854775807'.

`FLOAT(precision) [ZEROFILL]'
     A floating-point number. Cannot be unsigned.  `precision' can be
     `<=24' for a single precision floating point number and between 25
     and 53 for a double precision floating point number.  these types
     are like the `FLOAT' and `DOUBLE' types described immediately
     below. `FLOAT(X)' have the same ranges as the corresponding
     `FLOAT' and `DOUBLE' types, but the display size and number of
     decimals is undefined.

     In *MySQL* 3.23, this is a true floating point value.  In earlier
     *MySQL* versions, `FLOAT(precision)' always has 2 decimals.

     This syntax is provided for ODBC compatibility.

`FLOAT[(M,D)] [ZEROFILL]'
     A small (single-precision) floating-point number. Cannot be
     unsigned.  Allowable values are `-3.402823466E+38' to
     `-1.175494351E-38', `0' and `1.175494351E-38' to
     `3.402823466E+38'.  The M is the display width and D is the number
     of decimals.  `FLOAT' without an argument or with an argument of
     <= 24 stands for a single-precision floating point number.

`DOUBLE[(M,D)] [ZEROFILL]'
     A normal-size (double-precision) floating-point number. Cannot be
     unsigned. Allowable values are `-1.7976931348623157E+308' to
     `-2.2250738585072014E-308', `0' and `2.2250738585072014E-308' to
     `1.7976931348623157E+308'.  The M is the display width and D is
     the number of decimals.  `DOUBLE' without an argument or
     `FLOAT(X)' where 25 <= X <= 53 stands for a double-precision
     floating point number.

`DOUBLE PRECISION[(M,D)] [ZEROFILL]'
`REAL[(M,D)] [ZEROFILL]'
     These are synonyms for `DOUBLE'.

`DECIMAL[(M[,D])] [ZEROFILL]'
     An unpacked floating-point number.  Cannot be unsigned.  Behaves
     like a `CHAR' column: "unpacked" means the number is stored as a
     string, using one character for each digit of the value.  The
     decimal point, and, for negative numbers, the `-' sign is not
     counted in M.  If `D' is 0, values will have no decimal point or
     fractional part.  The maximum range of `DECIMAL' values is the
     same as for `DOUBLE', but the actual range for a given `DECIMAL'
     column may be constrained by the choice of `M' and `D'.

     If `D' is left out it's set to 0. If `M' is left out it's set to
     10.

     Note that in *MySQL* 3.22 the `M' argument includes the sign and
     the decimal point.

`NUMERIC(M,D) [ZEROFILL]'
     This is a synonym for `DECIMAL'.

`DATE'
     A date.  The supported range is `'1000-01-01'' to `'9999-12-31''.
     *MySQL* displays `DATE' values in `'YYYY-MM-DD'' format, but
     allows you to assign values to `DATE' columns using either strings
     or numbers.

`DATETIME'
     A date and time combination.  The supported range is `'1000-01-01
     00:00:00'' to `'9999-12-31 23:59:59''.  *MySQL* displays
     `DATETIME' values in `'YYYY-MM-DD HH:MM:SS'' format, but allows you
     to assign values to `DATETIME' columns using either strings or
     numbers.

`TIMESTAMP[(M)]'
     A timestamp.  The range is `'1970-01-01 00:00:00'' to sometime in
     the year `2037'.  *MySQL* displays `TIMESTAMP' values in
     `YYYYMMDDHHMMSS', `YYMMDDHHMMSS', `YYYYMMDD' or `YYMMDD' format,
     depending on whether `M' is `14' (or missing), `12', `8' or `6',
     but allows you to assign values to `TIMESTAMP' columns using
     either strings or numbers.  A `TIMESTAMP' column is useful for
     recording the date and time of an `INSERT' or `UPDATE' operation
     because it is automatically set to the date and time of the most
     recent operation if you don't give it a value yourself.  You can
     also set it to the current date and time by assigning it a `NULL'
     value.  *Note Date and time types::.

`TIME'
     A time.  The range is `'-838:59:59'' to `'838:59:59''.  *MySQL*
     displays `TIME' values in `'HH:MM:SS'' format, but allows you to
     assign values to `TIME' columns using either strings or numbers.

`YEAR[(2|4)]'
     A year in 2- or 4- digit formats (default is 4-digit).  The
     allowable values are `1901' to `2155', and `0000' in the 4 year
     format and 1970-2069 if you use the 2 digit format (70-69).
     *MySQL* displays `YEAR' values in `YYYY' format, but allows you to
     assign values to `YEAR' columns using either strings or numbers.
     (The `YEAR' type is new in *MySQL* 3.22.)

`[NATIONAL] CHAR(M) [BINARY]'
     A fixed-length string that is always right-padded with spaces to
     the specified length when stored. The range of `M' is 1 to 255
     characters.  Trailing spaces are removed when the value is
     retrieved. `CHAR' values are sorted and compared in
     case-insensitive fashion according to the default character set
     unless the `BINARY' keyword is given.

     `NATIONAL CHAR' (short form `NCHAR') is the ANSI SQL way to define
     that a CHAR column should use the default CHARACTER set.  This is
     default in `MySQL'.

     `CHAR' is a shorthand for `CHARACTER'.

     *MySQL* allows you to create a column of type `CHAR(0)'. This is
     mainly useful when you have to be compliant with some old
     applications that depend on the existence of a column but that do
     not actually use the value.  This is also quite nice when you need
     a column that only can take 2 values: A `CHAR(0)', that is not
     defined as `NOT NULL', will only occupy one bit and can only take
     2 values: `NULL' or `""'.

`[NATIONAL] VARCHAR(M) [BINARY]'
     A variable-length string.  Note: Trailing spaces are removed when
     the value is stored (this differs from the ANSI SQL
     specification). The range of `M' is 1 to 255 characters.
     `VARCHAR' values are sorted and compared in case-insensitive
     fashion unless the `BINARY' keyword is given. *Note Silent column
     changes::.

     `VARCHAR' is a shorthand for `CHARACTER VARYING'.

`TINYBLOB'
`TINYTEXT'
     A `BLOB' or `TEXT' column with a maximum length of 255 (2^8 - 1)
     characters. *Note Silent column changes::.

`BLOB'
`TEXT'
     A `BLOB' or `TEXT' column with a maximum length of 65535 (2^16 - 1)
     characters. *Note Silent column changes::.

`MEDIUMBLOB'
`MEDIUMTEXT'
     A `BLOB' or `TEXT' column with a maximum length of 16777215 (2^24
     - 1) characters. *Note Silent column changes::.

`LONGBLOB'
`LONGTEXT'
     A `BLOB' or `TEXT' column with a maximum length of 4294967295
     (2^32 - 1) characters. *Note Silent column changes::.

`ENUM('value1','value2',...)'
     An enumeration.  A string object that can have only one value,
     chosen from the list of values `'value1'', `'value2'', `...', or
     `NULL'.  An `ENUM' can have a maximum of 65535 distinct values.

`SET('value1','value2',...)'
     A set.  A string object that can have zero or more values, each of
     which must be chosen from the list of values `'value1'',
     `'value2'', `...' A `SET' can have a maximum of 64 members.

__data_types__

use constant H_MYSQL_VARIABLES => <<'__mysql_variables__';
Mysql Variables:
==============

*MySQL* supports thread specific variables with the `@variablename'
syntax.  A variable name may consist of alphanumeric characters from
the current character set and also `_', `$', and `.' . The default
character set is ISO-8859-1 Latin1; this may be changed with the
`--default-character-set' option to `mysqld'. *Note Character sets::.

Variables don't have to be initialized.  They contain `NULL' by default
and can store an integer, real or string value.  All variables for a
thread are automatically freed when the thread exits.

You can set a variable with the `SET' syntax:

     SET @variable= { integer expression | real expression | string expression }
     [,@variable= ...].

You can also set a variable in an expression with the `@variable:=expr'
syntax:

     select @t1:=(@t2:=1)+@t3:=4,@t1,@t2,@t3;
     +----------------------+------+------+------+
     | @t1:=(@t2:=1)+@t3:=4 | @t1  | @t2  | @t3  |
     +----------------------+------+------+------+
     |                    5 |    5 |    1 |    4 |
     +----------------------+------+------+------+

(We had to use the `:=' syntax here, because `=' was reserved for
comparisons.)

User variables may be used where expressions are allowed.  Note that
this does not currently include use in contexts where a number is
explicitly required, such as in the `LIMIT' clause of a `SELECT'
statement, or the `IGNORE number LINES' clause of a `LOAD DATA'
statement.

*NOTE:*  In a `SELECT' statement, each expression is only evaluated
when it's sent to the client.  This means that one can't in the
`HAVING', `GROUP BY' or `ORDER BY' clause refer to an expression that
involves variables that are set in the `SELECT' part.  For example, the
following statement will NOT work as expected:

     SELECT (@aa:=id) AS a, (@aa+3) AS b FROM table_name HAVING b=5;

The reason is that `@aa' will not contain the value of the current row,
but the value of `id' for the previous accepted row.

__mysql_variables__

use constant H_CREATE_DATABASE => <<'__create_database__';
CREATE DATABASE [IF NOT EXISTS] db_name

`CREATE DATABASE' creates a database with the given name.  An error
occurs if the database already exists and you didn't specify `IF NOT
EXISTS'.

Databases in *MySQL* are implemented as directories containing files
that correspond to tables in the database.  Because there are no tables
in a database when it is initially created, the `CREATE DATABASE'
statement only creates a directory under the *MySQL* data directory.

You can also create databases with `mysqladmin'. 
__create_database__

use constant CREATE_TABLE => <<'__create_table__';
CREATE [TEMPORARY] TABLE [IF NOT EXISTS] tbl_name
       [(create_definition,...)]
       [table_options] [select_statement]
     
   create_definition:
     col_name type [NOT NULL | NULL] [DEFAULT default_value] [AUTO_INCREMENT]
               [PRIMARY KEY] [reference_definition]
     or    PRIMARY KEY (index_col_name,...)
     or    KEY [index_name] (index_col_name,...)
     or    INDEX [index_name] (index_col_name,...)
     or    UNIQUE [INDEX] [index_name] (index_col_name,...)
     or    [CONSTRAINT symbol] FOREIGN KEY index_name (index_col_name,...)
               [reference_definition]
     or    CHECK (expr)
   
   type:
           TINYINT[(length)] [UNSIGNED] [ZEROFILL]
     or    SMALLINT[(length)] [UNSIGNED] [ZEROFILL]
     or    MEDIUMINT[(length)] [UNSIGNED] [ZEROFILL]
     or    INT[(length)] [UNSIGNED] [ZEROFILL]
     or    INTEGER[(length)] [UNSIGNED] [ZEROFILL]
     or    BIGINT[(length)] [UNSIGNED] [ZEROFILL]
     or    REAL[(length,decimals)] [UNSIGNED] [ZEROFILL]
     or    DOUBLE[(length,decimals)] [UNSIGNED] [ZEROFILL]
     or    FLOAT[(length,decimals)] [UNSIGNED] [ZEROFILL]
     or    DECIMAL(length,decimals) [UNSIGNED] [ZEROFILL]
     or    NUMERIC(length,decimals) [UNSIGNED] [ZEROFILL]
     or    CHAR(length) [BINARY]
     or    VARCHAR(length) [BINARY]
     or    DATE
     or    TIME
     or    TIMESTAMP
     or    DATETIME
     or    TINYBLOB
     or    BLOB
     or    MEDIUMBLOB
     or    LONGBLOB
     or    TINYTEXT
     or    TEXT
     or    MEDIUMTEXT
     or    LONGTEXT
     or    ENUM(value1,value2,value3,...)
     or    SET(value1,value2,value3,...)
   
   index_col_name:
           col_name [(length)]
   
   reference_definition:
           REFERENCES tbl_name [(index_col_name,...)]
                      [MATCH FULL | MATCH PARTIAL]
                      [ON DELETE reference_option]
                      [ON UPDATE reference_option]
   
   reference_option:
           RESTRICT | CASCADE | SET NULL | NO ACTION | SET DEFAULT
   
   table_options:
   	TYPE = {ISAM | MYISAM | HEAP}
   or	AUTO_INCREMENT = #
   or	AVG_ROW_LENGTH = #
   or	CHECKSUM = {0 | 1}
   or	COMMENT = "string"
   or	MAX_ROWS = #
   or	MIN_ROWS = #
   or	PACK_KEYS = {0 | 1}
   or	PASSWORD = "string"
   or	DELAY_KEY_WRITE = {0 | 1}
   or      ROW_FORMAT= { default | dynamic | static | compressed }
   or	RAID_TYPE= {1 | STRIPED | RAID0 } RAID_CHUNKS=#  RAID_CHUNKSIZE=#;
   
   select_statement:
   	[IGNORE | REPLACE] SELECT ...  (Some legal select statement)

`CREATE TABLE' creates a table with the given name in the current
database.  Rules for allowable table names are given in *Note Legal
names::.  An error occurs if there is no current database or if the
table already exists.

In *MySQL* 3.22 or later, the table name can be specified as
`db_name.tbl_name'.  This works whether or not there is a current
database.

In *MySQL* 3.23, you can use the `TEMPORARY' keyword when you create a
table.  A temporary table will automatically be deleted if a connection
dies and the name is per connection.  This means that two different
connections can both use the same temporary table name without
conflicting with each other or with an existing table of the same name.
(The existing table is hidden until the temporary table is deleted).

In *MySQL* 3.23 or later, you can use the keywords `IF NOT EXISTS' so
that an error does not occur if the table already exists.  Note that
there is no verification that the table structures are identical.

Each table `tbl_name' is represented by some files in the database
directory. In the case of MyISAM-type tables you will get:

*File*         *Purpose*
`tbl_name.frm' Table definition (form) file
`tbl_name.MYD' Data file
`tbl_name.MYI' Index file

For more information on the properties of the various column types, see
*Note Column types::.

   * If neither `NULL' nor `NOT NULL' is specified, the column is
     treated as though `NULL' had been specified.

   * An integer column may have the additional attribute
     `AUTO_INCREMENT'.  When you insert a value of `NULL' (recommended)
     or `0' into an `AUTO_INCREMENT' column, the column is set to
     `value+1', where `value' is the largest value for the column
     currently in the table.  `AUTO_INCREMENT' sequences begin with `1'.
     *Note `mysql_insert_id()': mysql_insert_id.

     If you delete the row containing the maximum value for an
     `AUTO_INCREMENT' column, the value will be reused with an ISAM
     table but not with a `MyISAM' table.  If you delete all rows in the
     table with `DELETE FROM table_name' (without a `WHERE') in
     `AUTOCOMMIT' mode, the sequence starts over for both table types.

     *Note:* There can be only one `AUTO_INCREMENT' column per table,
     and it must be indexed. *MySQL* 3.23 will also only work properly
     if the auto_increment column only has positive values. Inserting a
     negative number is regarded as inserting a very large positive
     number.  This is done to avoid precision problems when numbers
     'wrap' over from positive to negative and also to ensure that one
     doesn't accidently get a auto_increment column that contains 0.

     To make *MySQL* compatible with some ODBC applications, you can
     find the last inserted row with the following query:

          SELECT * FROM tbl_name WHERE auto_col IS NULL

   * `NULL' values are handled differently for `TIMESTAMP' columns than
     for other column types.  You cannot store a literal `NULL' in a
     `TIMESTAMP' column; setting the column to `NULL' sets it to the
     current date and time.  Because `TIMESTAMP' columns behave this
     way, the `NULL' and `NOT NULL' attributes do not apply in the
     normal way and are ignored if you specify them.

     On the other hand, to make it easier for *MySQL* clients to use
     `TIMESTAMP' columns, the server reports that such columns may be
     assigned `NULL' values (which is true), even though `TIMESTAMP'
     never actually will contain a `NULL' value.  You can see this when
     you use `DESCRIBE tbl_name' to get a description of your table.

     Note that setting a `TIMESTAMP' column to `0' is not the same as
     setting it to `NULL', because `0' is a valid `TIMESTAMP' value.

   * If no `DEFAULT' value is specified for a column, *MySQL*
     automatically assigns one.

     If the column may take `NULL' as a value, the default value is
     `NULL'.

     If the column is declared as `NOT NULL', the default value depends
     on the column type:

        - For numeric types other than those declared with the
          `AUTO_INCREMENT' attribute, the default is `0'.  For an
          `AUTO_INCREMENT' column, the default value is the next value
          in the sequence.

        - For date and time types other than `TIMESTAMP', the default
          is the appropriate "zero" value for the type.  For the first
          `TIMESTAMP' column in a table, the default value is the
          current date and time.  *Note Date and time types::.

        - For string types other than `ENUM', the default value is the
          empty string.  For `ENUM', the default is the first
          enumeration value.

   * `KEY' is a synonym for `INDEX'.

   * In *MySQL*, a `UNIQUE' key can have only distinct values. An error
     occurs if you try to add a new row with a key that matches an
     existing row.

   * A `PRIMARY KEY' is a unique `KEY' with the extra constraint that
     all key columns must be defined as `NOT NULL'.  In *MySQL* the key
     is named `PRIMARY'. A table can have only one `PRIMARY KEY'.  If
     you don't have a `PRIMARY KEY' and some applications ask for the
     `PRIMARY KEY' in your tables, *MySQL* will return the first
     `UNIQUE' key, which doesn't have any `NULL' columns, as the
     `PRIMARY KEY'.

   * A `PRIMARY KEY' can be a multiple-column index.  However, you
     cannot create a multiple-column index using the `PRIMARY KEY' key
     attibute in a column specification.  Doing so will mark only that
     single column as primary.  You must use the `PRIMARY
     KEY(index_col_name, ...)' syntax.

   * If the `PRIMARY' or `UNIQUE' key consists of only one column and
     this is of type integer, you can also refer to it as `_rowid' (new
     in 3.23.11).

   * If you don't assign a name to an index, the index will be assigned
     the same name as the first `index_col_name', with an optional
     suffix (`_2', `_3', `...') to make it unique.  You can see index
     names for a table using `SHOW INDEX FROM tbl_name'.  *Note `SHOW':
     SHOW.

   * Only the `MyISAM' table type supports indexes on columns that can
     have `NULL' values. In other cases you must declare such columns
     `NOT NULL' or an error results.

   * With `col_name(length)' syntax, you can specify an index which
     uses only a part of a `CHAR' or `VARCHAR' column. This can make
     the index file much smaller.  *Note Indexes::.

   * Only the `MyISAM' table type supports indexing on `BLOB' and
     `TEXT' columns.  When putting an index on a `BLOB' or `TEXT'
     column you MUST always specify the length of the index:
          CREATE TABLE test (blob_col BLOB, index(blob_col(10)));

   * When you use `ORDER BY' or `GROUP BY' with a `TEXT' or `BLOB'
     column, only the first `max_sort_length' bytes are used.  *Note
     `BLOB': BLOB.

   * The `FOREIGN KEY', `CHECK' and `REFERENCES' clauses don't actually
     do anything.  The syntax for them is provided only for
     compatibility, to make it easier to port code from other SQL
     servers and to run applications that create tables with references.
     *Note Missing functions::.

   * Each `NULL' column takes one bit extra, rounded up to the nearest
     byte.

   * The maximum record length in bytes can be calculated as follows:

          row length = 1
                       + (sum of column lengths)
                       + (number of NULL columns + 7)/8
                       + (number of variable-length columns)

   * The `table_options' and `SELECT' options is only implemented in
     *MySQL* 3.23 and above.

     The different table types are:

     ISAM          The original table handler. *Note ISAM::.
     MyISAM        The new binary portable table handler. *Note MyISAM::.
     HEAP          The data for this table is only stored in memory.
                   *Note HEAP::.
     BDB or        Transaction safe tables *Note BDB::.
     Berkeley_db   
     *Note Table types::.

     The other table options are used to optimize the behavior of the
     table. In most cases, you don't have to specify any of them.  The
     options work for all table types, if not otherwise indicated.

     `AUTO_INCREMENT'The next auto_increment value you want to set for
                   your table (MyISAM)
     `AVG_ROW_LENGTH'An approximation of the average row length for your
                   table. You only need to set this for tables with
                   variable size records.
     `CHECKSUM'    Set this to 1 if you want *MySQL* to maintain a
                   checksum for all rows (makes the table a little
                   slower to update but makes it easier to find
                   corrupted tables) (MyISAM)
     `COMMENT'     A 60 character comment for your table
     `MAX_ROWS'    Max number of rows you plan to store in the table
     `MIN_ROWS'    Minimum number of rows you plan to store in the table
     `PACK_KEYS'   Set this to 1 if you want to have smaller index. This
                   usually makes updates slower and reads faster
                   (MyISAM, ISAM).
     `PASSWORD'    Encrypt the `.frm' file with a password.  This option
                   doesn't do anything in the standard *MySQL* version.
     `DELAY_KEY_WRITE'Set this to 1 if want to delay key table updates
                   until the table is closed (MyISAM).
     `ROW_FORMAT'  Defines how the rows should be stored (for the
                   future).

     When you use a `MyISAM' table, *MySQL* uses the product of
     `max_rows * avg_row_length' to decide how big the resulting table
     will be.  If you don't specify any of the above options, the
     maximum size for a table will be 4G (or 2G if your operating
     systems only supports 2G tables).

     If you don't use `PACK_KEYS', the default is to only pack strings,
     not numbers.  If you use `PACK_KEYS=1', numbers will be packed as
     well.

     When packing binary number keys, *MySQL* will use prefix
     compression.  This means that you will only get a big benefit of
     this if you have many numbers that are the same.  Prefix
     compression means that every key needs one extra byte to indicate
     how many bytes of the previous key are the same for the next key
     (note that the pointer to the row is stored in
     high-byte-first-order directly after the key, to improve
     compression.  This means that if you have many equal keys on two
     rows in a row, all following 'same' keys will usually only take 2
     bytes (including the pointer to the row).  Compare this to the
     ordinary case where the following keys will take
     'storage_size_for_key' + pointer_size (usually 4).  On the other
     hand, if all keys are totally different, you will lose 1 byte per
     key, if the key isn't a key that can have `NULL' values (In this
     case the packed key length will be stored in the same byte that is
     used to mark if a key is `NULL').

   * If you specify a `SELECT' after the `CREATE STATEMENT', *MySQL*
     will create new fields for all elements in the `SELECT'.  For
     example:

          mysql> CREATE TABLE test (a int not null auto_increment,
                     primary key (a), key(b))
                     TYPE=HEAP SELECT b,c from test2;

     This will create a `HEAP' table with 3 columns.  Note that the
     table will automatically be deleted if any errors occur while
     copying data into the table.

   * The `RAID_TYPE' option will help you to break the 2G/4G limit on
     OSes that don't support big files. You can get also more speed
     from the I/O bottleneck by putting `RAID' directories on different
     physical disks.  `RAID_TYPE' will work on any OS, as long as you
     have configured *MySQL* with `--with-raid'.  For now the only
     allowed `RAID_TYPE' is `STRIPED' (`1' and `RAID0' are aliases for
     this).

     If you specify `RAID_TYPE=STRIPED' for a `MyISAM' table, `MyISAM'
     will create `RAID_CHUNKS' sub-directories named 00, 01, 02 in the
     database directory.  In each of these directories `MyISAM' will
     create an `table_name.MYD'.  When writing data to the data file,
     the `RAID' handler will map the first `RAID_CHUNKSIZE' *1024 bytes
     to the first file, the next `RAID_CHUNKSIZE' *1024 bytes to the
     next file and so on.

__create_table__

use constant H_DROP_DATABASE => <<'__drop_database__';
DROP DATABASE [IF EXISTS] db_name

`DROP DATABASE' drops all tables in the database and deletes the
database.  *Be VERY careful with this command!*

`DROP DATABASE' returns the number of files that were removed from the
database directory.  Normally, this is three times the number of tables,
because each table corresponds to a `.MYD' file, a `.MYI' file and a
`.frm' file.

In *MySQL* 3.22 or later, you can use the keywords `IF EXISTS' to
prevent an error from occurring if the database doesn't exist.

You can also drop databases with `mysqladmin'.
__drop_database__

use constant H_ALTER_TABLE => <<'__alter_table__';
ALTER [IGNORE] TABLE tbl_name alter_spec [, alter_spec ...]
     
     alter_specification:
             ADD [COLUMN] create_definition [FIRST | AFTER column_name ]
       or    ADD [COLUMN] (create_definition, create_definition,...)
       or    ADD INDEX [index_name] (index_col_name,...)
       or    ADD PRIMARY KEY (index_col_name,...)
       or    ADD UNIQUE [index_name] (index_col_name,...)
       or    ALTER [COLUMN] col_name {SET DEFAULT literal | DROP DEFAULT}
       or    CHANGE [COLUMN] old_col_name create_definition
       or    MODIFY [COLUMN] create_definition
       or    DROP [COLUMN] col_name
       or    DROP PRIMARY KEY
       or    DROP INDEX index_name
       or    RENAME [AS] new_tbl_name
       or    table_options

`ALTER TABLE' allows you to change the structure of an existing table.
For example, you can add or delete columns, create or destroy indexes,
change the type of existing columns, or rename columns or the table
itself.  You can also change the comment for the table and type of the
table.  *Note `CREATE TABLE': CREATE TABLE.

If you use `ALTER TABLE' to change a column specification but `DESCRIBE
tbl_name' indicates that your column was not changed, it is possible
that *MySQL* ignored your modification for one of the reasons described
in *Note Silent column changes::.  For example, if you try to change a
`VARCHAR' column to `CHAR', *MySQL* will still use `VARCHAR' if the
table contains other variable-length columns.

`ALTER TABLE' works by making a temporary copy of the original table.
The alteration is performed on the copy, then the original table is
deleted and the new one is renamed. This is done in such a way that all
updates are automatically redirected to the new table without any
failed updates. While `ALTER TABLE' is executing, the original table is
readable by other clients. Updates and writes to the table are stalled
until the new table is ready.

   * To use `ALTER TABLE', you need *select*, *insert*, *delete*,
     *update*, *create* and *drop* privileges on the table.

   * `IGNORE' is a *MySQL* extension to ANSI SQL92.  It controls how
     `ALTER TABLE' works if there are duplicates on unique keys in the
     new table.  If `IGNORE' isn't specified, the copy is aborted and
     rolled back.  If `IGNORE' is specified, then for rows with
     duplicates on a unique key, only the first row is used; the others
     are deleted.

   * You can issue multiple `ADD', `ALTER', `DROP' and `CHANGE' clauses
     in a single `ALTER TABLE' statement. This is a *MySQL* extension
     to ANSI SQL92, which allows only one of each clause per `ALTER
     TABLE' statement.

   * `CHANGE col_name', `DROP col_name' and `DROP INDEX' are *MySQL*
     extensions to ANSI SQL92.

   * `MODIFY' is an Oracle extension to `ALTER TABLE'.

   * The optional word `COLUMN' is a pure noise word and can be omitted.

   * If you use `ALTER TABLE tbl_name RENAME AS new_name' without any
     other options, *MySQL* simply renames the files that correspond to
     the table `tbl_name'.  There is no need to create the temporary
     table.

   * `create_definition' clauses use the same syntax for `ADD' and
     `CHANGE' as for `CREATE TABLE'.  Note that this syntax includes
     the column name, not just the column type.  *Note `CREATE TABLE':
     CREATE TABLE.

   * You can rename a column using a `CHANGE old_col_name
     create_definition' clause.  To do so, specify the old and new
     column names and the type that the column currently has.  For
     example, to rename an `INTEGER' column from `a' to `b', you can do
     this:

          mysql> ALTER TABLE t1 CHANGE a b INTEGER;

     If you want to change a column's type but not the name, `CHANGE'
     syntax still requires two column names even if they are the same.
     For example:

          mysql> ALTER TABLE t1 CHANGE b b BIGINT NOT NULL;

     However, as of *MySQL* 3.22.16a, you can also use `MODIFY' to
     change a column's type without renaming it:

          mysql> ALTER TABLE t1 MODIFY b BIGINT NOT NULL;

   * If you use `CHANGE' or `MODIFY' to shorten a column for which an
     index exists on part of the column (for instance, if you have an
     index on the first 10 characters of a `VARCHAR' column), you
     cannot make the column shorter than the number of characters that
     are indexed.

   * When you change a column type using `CHANGE' or `MODIFY', *MySQL*
     tries to convert data to the new type as well as possible.

   * In *MySQL* 3.22 or later, you can use `FIRST' or `ADD ...  AFTER
     col_name' to add a column at a specific position within a table
     row.  The default is to add the column last.

   * `ALTER COLUMN' specifies a new default value for a column or
     removes the old default value.  If the old default is removed and
     the column can be `NULL', the new default is `NULL'. If the column
     cannot be `NULL', *MySQL* assigns a default value.  Default value
     assignment is described in *Note `CREATE TABLE': CREATE TABLE.

   * `DROP INDEX' removes an index. This is a *MySQL* extension to ANSI
     SQL92.

   * If columns are dropped from a table, the columns are also removed
     from any index of which they are a part.  If all columns that make
     up an index are dropped, the index is dropped as well.

   * `DROP PRIMARY KEY' drops the primary index. If no such index
     exists, it drops the first `UNIQUE' index in the table.  (*MySQL*
     marks the first `UNIQUE' key as the `PRIMARY KEY' if no `PRIMARY
     KEY' was specified explicitly.)

   * With the C API function `mysql_info()', you can find out how many
     records were copied, and (when `IGNORE' is used) how many records
     were deleted due to duplication of unique key values.

   * The `FOREIGN KEY', `CHECK' and `REFERENCES' clauses don't actually
     do anything.  The syntax for them is provided only for
     compatibility, to make it easier to port code from other SQL
     servers and to run applications that create tables with references.
     *Note Missing functions::.

Here is an example that shows some of the uses of `ALTER TABLE'.  We
begin with a table `t1' that is created as shown below:

     mysql> CREATE TABLE t1 (a INTEGER,b CHAR(10));

To rename the table from `t1' to `t2':

     mysql> ALTER TABLE t1 RENAME t2;

To change column `a' from `INTEGER' to `TINYINT NOT NULL' (leaving the
name the same), and to change column `b' from `CHAR(10)' to `CHAR(20)'
as well as renaming it from `b' to `c':

     mysql> ALTER TABLE t2 MODIFY a TINYINT NOT NULL, CHANGE b c CHAR(20);

To add a new `TIMESTAMP' column named `d':

     mysql> ALTER TABLE t2 ADD d TIMESTAMP;

To add an index on column `d', and make column `a' the primary key:

     mysql> ALTER TABLE t2 ADD INDEX (d), ADD PRIMARY KEY (a);

To remove column `c':

     mysql> ALTER TABLE t2 DROP COLUMN c;

To add a new `AUTO_INCREMENT' integer column named `c':

     mysql> ALTER TABLE t2 ADD c INT UNSIGNED NOT NULL AUTO_INCREMENT,
                ADD INDEX (c);

Note that we indexed `c', because `AUTO_INCREMENT' columns must be
indexed, and also that we declare `c' as `NOT NULL', because indexed
columns cannot be `NULL'.

When you add an `AUTO_INCREMENT' column, column values are filled in
with sequence numbers for you automatically.

Problems with `ALTER TABLE'.
============================

If `ALTER TABLE' dies with an error like this:

     Error on rename of './database/name.frm' to './database/B-a.frm' (Errcode: 17)

The problem may be that *MySQL* has crashed in a previous `ALTER TABLE'
and there is an old table named `A-something' or `B-something' lying
around.  In this case, go to the *MySQL* data directory and delete all
files that have names starting with `A-' or `B-'.  (You may want to
move them elsewhere instead of deleting them).

`ALTER TABLE' works the following way:

   * Create a new table named `A-xxx' with the requested changes.

   * All rows from the old table are copied to `A-xxx'.

   * The old table is renamed `B-xxx'.

   * `A-xxx' is renamed to your old table name.

   * `B-xxx' is deleted.

If something goes wrong with the renaming operation, *MySQL* tries to
undo the changes.  If something goes seriously wrong (this shouldn't
happen, of course), *MySQL* may leave the old table as `B-xxx' but a
simple rename should get your data back.

Caveat Administrator

__alter_table__

use constant H_DROP_TABLE => <<'__drop_table__';
DROP TABLE [IF EXISTS] tbl_name [, tbl_name,...]

`DROP TABLE' removes one or more tables. All table data and the table
definition are _removed_, so *be careful* with this command!

In *MySQL* 3.22 or later, you can use the keywords `IF EXISTS' to
prevent an error from occurring for tables that don't exist.
__drop_table__

use constant H_OPTIMIZE_TABLE => <<'__optimize_table__';
OPTIMIZE TABLE tbl_name

`OPTIMIZE TABLE' should be used if you have deleted a large part of a
table or if you have made many changes to a table with variable-length
rows (tables that have `VARCHAR', `BLOB' or `TEXT' columns).  Deleted
records are maintained in a linked list and subsequent `INSERT'
operations reuse old record positions. You can use `OPTIMIZE TABLE' to
reclaim the unused space.

`OPTIMIZE TABLE' works by making a temporary copy of the original
table; The old table is copied to the new table (without the unused
rows), then the original table is deleted and the new one is renamed.
While `OPTIMIZE TABLE' is executing, the original table is readable by
other clients. Updates and writes to the table are stalled until the
new table is ready. This is done in such a way that all updates are
automatically redirected to the new table without any failed updates.
__optimize_table__

use constant H_CHECK_TABLE => <<'__check_table__';
CHECK TABLE tbl_name[,tbl_name...] [TYPE = QUICK]

Check the table(s) for errors. The command returns a table with the
following columns:

Table                     Table name
Op                        Always 'check'
Msg_type                  One of `status', `error', `info' or `warning'.
Msg_text                  The message.

Note that you can get many rows of information for each checked table.
The last one row will be of `Msg_type status' and should normally be
`OK'.  If you don't get `OK', you should normally run a repair of the
table. *Note Table maintenance::.

If `TYPE=QUICK' is given then *MySQL* will not scan the rows for table
with fixed size records.

`CHECK TABLE' only works on `MyISAM' tables and is the same things as
running `myisamchk -m table_name' on the table.

__check_table__

use constant ANALYZE_TABLE => <<'__analyze_table__';
ANALYZE TABLE tbl_name[,tbl_name...]

Analyze and store the key distribution for the table.  During the
analyze the table is locked with a read lock.  This is equivalent of
running `myisamchk -a' on the table.

*MySQL* uses the stored key distribution to decide in which order
tables should be joined when one does a join on something else than a
constant.

The command returns a table with the following columns:

Table                     Table name
Op                        Always 'analyze
Msg_type                  One of `status', `error', `info' or `warning'.
Msg_text                  The message.

You can check the stored key distribution with the `SHOW INDEX' command.

__analyze_table__

use constant H_REPAIR_TABLE => <<'__repair_table__';
ANALYZE TABLE tbl_name[,tbl_name...]

Analyze and store the key distribution for the table.  During the
analyze the table is locked with a read lock.  This is equivalent of
running `myisamchk -a' on the table.

*MySQL* uses the stored key distribution to decide in which order
tables should be joined when one does a join on something else than a
constant.

The command returns a table with the following columns:

Table                     Table name
Op                        Always 'analyze
Msg_type                  One of `status', `error', `info' or `warning'.
Msg_text                  The message.

You can check the stored key distribution with the `SHOW INDEX' command.
__repair_table__
  
use constant H_DELETE => <<'__delete__';
DELETE [LOW_PRIORITY] FROM tbl_name
       [WHERE where_definition] [LIMIT rows]

`DELETE' deletes rows from `tbl_name' that satisfy the condition given
by `where_definition', and returns the number of records deleted.

If you issue a `DELETE' with no `WHERE' clause, all rows are deleted.
If you do this in `AUTOCOMMIT' mode, *MySQL* does this by recreating
the table as an empty table, which is much faster than deleting each
row.  In this case, `DELETE' returns zero as the number of affected
records. (*MySQL* can't return the number of rows that were actually
deleted, because the recreate is done without opening the data files.
As long as the table definition file `tbl_name.frm' is valid, the table
can be recreated this way, even if the data or index files have become
corrupted.).

If you really want to know how many records are deleted when you are
deleting all rows, and are willing to suffer a speed penalty, you can
use a `DELETE' statement of this form:

     mysql> DELETE FROM tbl_name WHERE 1>0;

Note that this is MUCH slower than `DELETE FROM tbl_name' with no
`WHERE' clause, because it deletes rows one at a time.

If you specify the keyword `LOW_PRIORITY', execution of the `DELETE' is
delayed until no other clients are reading from the table.

Deleted records are maintained in a linked list and subsequent `INSERT'
operations reuse old record positions. To reclaim unused space and
reduce file sizes, use the `OPTIMIZE TABLE' statement or the `myisamchk'
utility to reorganize tables.  `OPTIMIZE TABLE' is easier, but
`myisamchk' is faster.

The *MySQL*-specific `LIMIT rows' option to `DELETE' tells the server
the maximum number of rows to be deleted before control is returned to
the client.  This can be used to ensure that a specific `DELETE'
command doesn't take too much time.  You can simply repeat the `DELETE'
command until the number of affected rows is less than the `LIMIT'
value.

__delete__

use constant H_SELECT => <<'__select__';
SELECT [STRAIGHT_JOIN] [SQL_SMALL_RESULT] [SQL_BIG_RESULT] [SQL_BUFFER_RESULT]
        [HIGH_PRIORITY]
        [DISTINCT | DISTINCTROW | ALL]
     select_expression,...
     [INTO {OUTFILE | DUMPFILE} 'file_name' export_options]
     [FROM table_references
         [WHERE where_definition]
         [GROUP BY {unsigned_integer | col_name | formula}]
         [HAVING where_definition]
         [ORDER BY {unsigned_integer | col_name | formula} [ASC | DESC] ,...]
         [LIMIT [offset,] rows]
         [PROCEDURE procedure_name] ]

`SELECT' is used to retrieve rows selected from one or more tables.
`select_expression' indicates the columns you want to retrieve.

`SELECT' may also be used to retrieve rows computed without reference to
any table.  For example:

     mysql> SELECT 1 + 1;
              -> 2

All keywords used must be given in exactly the order shown above. For
example, a `HAVING' clause must come after any `GROUP BY' clause and
before any `ORDER BY' clause.

   * A `SELECT' expression may be given an alias using `AS'. The alias
     is used as the expression's column name and can be used with
     `ORDER BY' or `HAVING' clauses.  For example:

          mysql> select concat(last_name,', ',first_name) AS full_name
              from mytable ORDER BY full_name;

   * The `FROM table_references' clause indicates the tables from which
     to retrieve rows.  If you name more than one table, you are
     performing a join.  For information on join syntax, see *Note
     `JOIN': JOIN.

   * You can refer to a column as `col_name', `tbl_name.col_name' or
     `db_name.tbl_name.col_name'.  You need not specify a `tbl_name' or
     `db_name.tbl_name' prefix for a column reference in a `SELECT'
     statement unless the reference would be ambiguous.  See *Note
     Legal names::, for examples of ambiguity that require the more
     explicit column reference forms.

   * A table reference may be aliased using `tbl_name [AS] alias_name'.

          mysql> select t1.name, t2.salary from employee AS t1, info AS t2
                     where t1.name = t2.name;
          mysql> select t1.name, t2.salary from employee t1, info t2
                     where t1.name = t2.name;

   * Columns selected for output may be referred to in `ORDER BY' and
     `GROUP BY' clauses using column names, column aliases or column
     positions.  Column positions begin with 1.

          mysql> select college, region, seed from tournament
                     ORDER BY region, seed;
          mysql> select college, region AS r, seed AS s from tournament
                     ORDER BY r, s;
          mysql> select college, region, seed from tournament
                     ORDER BY 2, 3;

     To sort in reverse order, add the `DESC' (descending) keyword to
     the name of the column in the `ORDER BY' clause that you are
     sorting by.  The default is ascending order; this may be specified
     explicitly using the `ASC' keyword.

   * The `HAVING' clause can refer to any column or alias named in the
     `select_expression'. It is applied last, just before items are
     sent to the client, with no optimization. Don't use `HAVING' for
     items that should be in the `WHERE' clause.  For example, do not
     write this:

          mysql> select col_name from tbl_name HAVING col_name > 0;

     Write this instead:

          mysql> select col_name from tbl_name WHERE col_name > 0;

     In *MySQL* 3.22.5 or later, you can also write queries like this:

          mysql> select user,max(salary) from users
                     group by user HAVING max(salary)>10;

     In older *MySQL* versions, you can write this instead:

          mysql> select user,max(salary) AS sum from users
                     group by user HAVING sum>10;

   * `SQL_SMALL_RESULT', `SQL_BIG_RESULT', `SQL_BUFFER_RESULT',
     `STRAIGHT_JOIN' and `HIGH_PRIORITY' are *MySQL* extensions to ANSI
     SQL92.

   * `HIGH_PRIORITY' will give the `SELECT' higher priority than a
     statement that updates a table.  You should only use this for
     queries that are very fast and must be done at once.  A `SELECT
     HIGH_PRIORITY' query will run if the table is locked for read even
     if there is an update statement that is waiting for the table to
     be free.

   * `SQL_BIG_RESULT' can be used with `GROUP BY' or `DISTINCT' to tell
     the optimizer that the result set will have many rows.  In this
     case, *MySQL* will directly use disk based temporary tables if
     needed.  *MySQL* will also in this case prefer sorting to doing a
     temporary table with a key on the `GROUP BY' elements.

   * `SQL_BUFFER_RESULT' will put force the result to be put into a
     temporary table. This will help *MySQL* free the table locks early
     and will help in cases where it takes a long time to send the
     result set to the client.

   * `SQL_SMALL_RESULT', a *MySQL*-specific option, can be used with
     `GROUP BY' or `DISTINCT' to tell the optimizer that the result set
     will be small.  In this case, *MySQL* will use fast temporary
     tables to store the resulting table instead of using sorting. In
     *MySQL* 3.23 this shouldn't normally be needed.

   * `STRAIGHT_JOIN' forces the optimizer to join the tables in the
     order in which they are listed in the `FROM' clause. You can use
     this to speed up a query if the optimizer joins the tables in
     non-optimal order.  *Note `EXPLAIN': EXPLAIN.

   * The `LIMIT' clause can be used to constrain the number of rows
     returned by the `SELECT' statement.  `LIMIT' takes one or two
     numeric arguments.

     If two arguments are given, the first specifies the offset of the
     first row to return, the second specifies the maximum number of
     rows to return.  The offset of the initial row is 0 (not 1).

          mysql> select * from table LIMIT 5,10;  # Retrieve rows 6-15

     If one argument is given, it indicates the maximum number of rows
     to return.

          mysql> select * from table LIMIT 5;     # Retrieve first 5 rows

     In other words, `LIMIT n' is equivalent to `LIMIT 0,n'.

   * The `SELECT ... INTO OUTFILE 'file_name'' form of `SELECT' writes
     the selected rows to a file. The file is created on the server
     host, and cannot already exist (among other things, this prevents
     database tables and files such as `/etc/passwd' from being
     destroyed).  You must have the *file* privilege on the server host
     to use this form of `SELECT'.

     `SELECT ...  INTO OUTFILE' is the complement of `LOAD DATA
     INFILE'; the syntax for the `export_options' part of the statement
     consists of the same `FIELDS' and `LINES' clauses that are used
     with the `LOAD DATA INFILE' statement.  *Note `LOAD DATA': LOAD
     DATA.

     In the resulting text file, only the following characters are
     escaped by the `ESCAPED BY' character:

        * The `ESCAPED BY' character

        * The first character in `FIELDS TERMINATED BY'

        * The first character in `LINES TERMINATED BY'

     Additionally, `ASCII 0' is converted to `ESCAPED BY' followed by 0
     (`ASCII 48').

     The reason for the above is that you MUST escape any `FIELDS
     TERMINATED BY', `ESCAPED BY' or `LINES TERMINATED BY' characters
     to reliably be able to read the file back. `ASCII 0' is escaped to
     make it easier to view with some pagers.

     As the resulting file doesn't have to conform to the SQL syntax,
     nothing else need be escaped.

If you use `INTO DUMPFILE' instead of `INTO OUTFILE' *MySQL* will only
write one row into the file, without any column or line terminations
and without any escaping.  This is useful if you want to store a blob
in a file.

__select__

use constant H_JOIN => <<'__join__';

*MySQL* supports the following `JOIN' syntaxes for use in `SELECT'
statements:

     table_reference, table_reference
     table_reference [CROSS] JOIN table_reference
     table_reference INNER JOIN table_reference join_condition
     table_reference STRAIGHT_JOIN table_reference
     table_reference LEFT [OUTER] JOIN table_reference join_condition
     table_reference LEFT [OUTER] JOIN table_reference
     table_reference NATURAL [LEFT [OUTER]] JOIN table_reference
     { oj table_reference LEFT OUTER JOIN table_reference ON conditional_expr }

Where `table_reference' is defined as
     table_name [[AS] alias] [USE INDEX (key_list)] [IGNORE INDEX (key_list)]

and `join_condition' is defined as

     ON conditional_expr |
     USING (column_list)

Note that in version before 3.23.16 the `INNER JOIN' didn't take a join
condition!

The last `LEFT OUTER JOIN' syntax shown above exists only for
compatibility with ODBC.

   * A table reference may be aliased using `tbl_name AS alias_name' or
     `tbl_name alias_name'.

          mysql> select t1.name, t2.salary from employee AS t1, info AS t2
                     where t1.name = t2.name;

   * `INNER JOIN' and `,' (comma) are semantically equivalent.  Both do
     a full join between the tables used.  Normally, you specify how
     the tables should be linked in the `WHERE' condition.

   * The `ON' conditional is any conditional of the form that may be
     used in a `WHERE' clause.

   * If there is no matching record for the right table in the `ON' or
     `USING' part in a `LEFT JOIN', a row with all columns set to
     `NULL' is used for the right table.  You can use this fact to find
     records in a table that have no counterpart in another table:

          mysql> select table1.* from table1
                     LEFT JOIN table2 ON table1.id=table2.id
                     where table2.id is NULL;

     This example finds all rows in `table1' with an `id' value that is
     not present in `table2' (i.e., all rows in `table1' with no
     corresponding row in `table2').  This assumes that `table2.id' is
     declared `NOT NULL', of course.

   * The `USING' `(column_list)' clause names a list of columns that
     must exist in both tables.  A `USING' clause such as:

          A LEFT JOIN B USING (C1,C2,C3,...)

     is defined to be semantically identical to an `ON' expression like
     this:

          A.C1=B.C1 AND A.C2=B.C2 AND A.C3=B.C3,...

   * The `NATURAL [LEFT] JOIN' of two tables is defined to be
     semantically equivalent to a `INNER JOIN' or a `LEFT JOIN' with a
     `USING' clause that names all columns that exist in both tables.

   * `STRAIGHT_JOIN' is identical to `JOIN', except that the left table
     is always read before the right table. This can be used for those
     (few) cases where the join optimizer puts the tables in the wrong
     order.

   * As of *MySQL* 3.23.12, you can give hints about which index
     *MySQL* should use when retrieving information from a table. This
     is useful if `EXPLAIN' shows that *MySQL* is using the wrong
     index.  By specifying `USE INDEX (key_list)', you can tell *MySQL*
     to use only one of the specified indexes to find rows in the
     table.  The alternative syntax `IGNORE INDEX (key_list)' can be
     used to tell *MySQL* to not use some particular index.

Some examples:

  mysql> select * from table1,table2 where table1.id=table2.id;
  mysql> select * from table1 LEFT JOIN table2 ON table1.id=table2.id;
  mysql> select * from table1 LEFT JOIN table2 USING (id);
  mysql> select * from table1 LEFT JOIN table2 ON table1.id=table2.id
           LEFT JOIN table3 ON table2.id=table3.id;
  mysql> select * from table1 USE INDEX (key1,key2) WHERE key1=1 and key2=2 AND
           key3=3;
  mysql> select * from table1 IGNORE INDEX (key3) WHERE key1=1 and key2=2 AND
           key3=3;

__join__


use constant H_INSERT => <<'__insert__';
INSERT [LOW_PRIORITY | DELAYED] [IGNORE]
       [INTO] tbl_name [(col_name,...)]
        VALUES (expression,...),(...),...

INSERT [LOW_PRIORITY | DELAYED] [IGNORE]
       [INTO] tbl_name [(col_name,...)]
       SELECT ...

INSERT [LOW_PRIORITY | DELAYED] [IGNORE]
       [INTO] tbl_name
       SET col_name=expression, col_name=expression, ...

`INSERT' inserts new rows into an existing table.  The `INSERT ...
VALUES' form of the statement inserts rows based on explicitly-specified
values.  The `INSERT ... SELECT' form inserts rows selected from another
table or tables.  The `INSERT ... VALUES' form with multiple value lists
is supported in *MySQL* 3.22.5 or later.  The `col_name=expression'
syntax is supported in *MySQL* 3.22.10 or later.

`tbl_name' is the table into which rows should be inserted.  The column
name list or the `SET' clause indicates which columns the statement
specifies values for.

   * If you specify no column list for `INSERT ... VALUES' or `INSERT
     ... SELECT', values for all columns must be provided in the
     `VALUES()' list or by the `SELECT'.  If you don't know the order of
     the columns in the table, use `DESCRIBE tbl_name' to find out.

   * Any column not explicitly given a value is set to its default
     value.  For example, if you specify a column list that doesn't
     name all the columns in the table, unnamed columns are set to
     their default values.  Default value assignment is described in
     *Note `CREATE TABLE': CREATE TABLE.

   * An `expression' may refer to any column that was set earlier in a
     value list.  For example, you can say this:

          mysql> INSERT INTO tbl_name (col1,col2) VALUES(15,col1*2);

     But not this:

          mysql> INSERT INTO tbl_name (col1,col2) VALUES(col2*2,15);

   * If you specify the keyword `LOW_PRIORITY', execution of the
     `INSERT' is delayed until no other clients are reading from the
     table.  In this case the client has to wait until the insert
     statement is completed, which may take a long time if the table is
     in heavy use. This is in contrast to `INSERT DELAYED' which lets
     the client continue at once.

   * If you specify the keyword `IGNORE' in an `INSERT' with many value
     rows, any rows which duplicate an existing `PRIMARY' or `UNIQUE'
     key in the table are ignored and are not inserted.  If you do not
     specify `IGNORE', the insert is aborted if there is any row that
     duplicates an existing key value.  You can check with the C API
     function `mysql_info()' how many rows were inserted into the table.

   * If *MySQL* was configured using the `DONT_USE_DEFAULT_FIELDS'
     option, `INSERT' statements generate an error unless you explicitly
     specify values for all columns that require a non-`NULL' value.
     *Note `configure' options: configure options.

   * The following conditions hold for a `INSERT INTO ... SELECT'
     statement:

        - The query cannot contain an `ORDER BY' clause.

        - The target table of the `INSERT' statement cannot appear in
          the `FROM' clause of the `SELECT' part of the query, because
          it's forbidden in ANSI SQL to `SELECT' from the same table
          into which you are `INSERT'ing.  (The problem is that the
          `SELECT' possibly would find records that were inserted
          earlier during the same run.  When using sub-select clauses,
          the situation could easily be very confusing!)

        - `AUTO_INCREMENT' columns work as usual.

If you use `INSERT ... SELECT' or a `INSERT ... VALUES' statement with
multiple value lists, you can use the C API function `mysql_info()' to
get information about the query.  The format of the information string
is shown below:

     Records: 100 Duplicates: 0 Warnings: 0

`Duplicates' indicates the number of rows that couldn't be inserted
because they would duplicate some existing unique index value.
`Warnings' indicates the number of attempts to insert column values that
were problematic in some way. Warnings can occur under any of the
following conditions:

   * Inserting `NULL' into a column that has been declared `NOT NULL'.
     The column is set to its default value.

   * Setting a numeric column to a value that lies outside the column's
     range.  The value is clipped to the appropriate endpoint of the
     range.

   * Setting a numeric column to a value such as `'10.34 a''.  The
     trailing garbage is stripped and the remaining numeric part is
     inserted.  If the value doesn't make sense as a number at all, the
     column is set to `0'.

   * Inserting a string into a `CHAR', `VARCHAR', `TEXT' or `BLOB'
     column that exceeds the column's maximum length.  The value is
     truncated to the column's maximum length.

   * Inserting a value into a date or time column that is illegal for
     the column type.  The column is set to the appropriate "zero"
     value for the type.

The `DELAYED' option for the `INSERT' statement is a *MySQL*-specific
option that is very useful if you have clients that can't wait for the
`INSERT' to complete.  This is a common problem when you use *MySQL*
for logging and you also periodically run `SELECT' statements that take
a long time to complete.  `DELAYED' was introduced in *MySQL* 3.22.15.
It is a *MySQL* extension to ANSI SQL92.

When you use `INSERT DELAYED', the client will get an ok at once and
the row will be inserted when the table is not in use by any other
thread.

Another major benefit of using `INSERT DELAYED' is that inserts from
many clients are bundled together and written in one block. This is much
faster than doing many separate inserts.

Note that currently the queued rows are only stored in memory until
they are inserted into the table.  This means that if you kill `mysqld'
the hard way (`kill -9') or if `mysqld' dies unexpectedly, any queued
rows that weren't written to disk are lost!

The following describes in detail what happens when you use the
`DELAYED' option to `INSERT' or `REPLACE'.  In this description, the
"thread" is the thread that received an `INSERT DELAYED' command and
"handler" is the thread that handles all `INSERT DELAYED' statements
for a particular table.

   * When a thread executes a `DELAYED' statement for a table, a handler
     thread is created to process all `DELAYED' statements for the
     table, if no such handler already exists.

   * The thread checks whether or not the handler has acquired a
     `DELAYED' lock already; if not, it tells the handler thread to do
     so.  The `DELAYED' lock can be obtained even if other threads have
     a `READ' or `WRITE' lock on the table.  However, the handler will
     wait for all `ALTER TABLE' locks or `FLUSH TABLES' to ensure that
     the table structure is up to date.

   * The thread executes the `INSERT' statement but instead of writing
     the row to the table it puts a copy of the final row into a queue
     that is managed by the handler thread. Any syntax errors are
     noticed by the thread and reported to the client program.

   * The client can't report the number of duplicates or the
     `AUTO_INCREMENT' value for the resulting row; it can't obtain them
     from the server, because the `INSERT' returns before the insert
     operation has been completed.  If you use the C API, the
     `mysql_info()' function doesn't return anything meaningful, for
     the same reason.

   * The update log is updated by the handler thread when the row is
     inserted into the table.  In case of multiple-row inserts, the
     update log is updated when the first row is inserted.

   * After every `delayed_insert_limit' rows are written, the handler
     checks whether or not any `SELECT' statements are still pending.
     If so, it allows these to execute before continuing.

   * When the handler has no more rows in its queue, the table is
     unlocked.  If no new `INSERT DELAYED' commands are received within
     `delayed_insert_timeout' seconds, the handler terminates.

   * If more than `delayed_queue_size' rows are pending already in a
     specific handler queue, the thread waits until there is room in
     the queue.  This is useful to ensure that the `mysqld' server
     doesn't use all memory for the delayed memory queue.

   * The handler thread will show up in the *MySQL* process list with
     `delayed_insert' in the `Command' column.  It will be killed if
     you execute a `FLUSH TABLES' command or kill it with `KILL
     thread_id'. However, it will first store all queued rows into the
     table before exiting. During this time it will not accept any new
     `INSERT' commands from another thread. If you execute an `INSERT
     DELAYED' command after this, a new handler thread will be created.

   * Note that the above means that `INSERT DELAYED' commands have
     higher priority than normal `INSERT' commands if there is an
     `INSERT DELAYED' handler already running!  Other update commands
     will have to wait until the `INSERT DELAYED' queue is empty,
     someone kills the handler thread (with `KILL thread_id') or
     someone executes `FLUSH TABLES'.

   * The following status variables provide information about `INSERT
     DELAYED' commands:

     `Delayed_insert_threads'Number of handler threads
     `Delayed_writes'        Number of rows written with `INSERT DELAYED'
     `Not_flushed_delayed_rows'Number of rows waiting to be written

     You can view these variables by issuing a `SHOW STATUS' statement
     or by executing a `mysqladmin extended-status' command.

Note that `INSERT DELAYED' is slower than a normal INSERT if the table
is not in use.  There is also the additional overhead for the server to
handle a separate thread for each table on which you use `INSERT
DELAYED'.  This means that you should only use `INSERT DELAYED' when
you are really sure you need it!

__insert__

use constant H_REPLACE => <<'__replace__';
REPLACE [LOW_PRIORITY | DELAYED]
        [INTO] tbl_name [(col_name,...)]
        VALUES (expression,...)
  
REPLACE [LOW_PRIORITY | DELAYED]
        [INTO] tbl_name [(col_name,...)]
        SELECT ...

REPLACE [LOW_PRIORITY | DELAYED]
        [INTO] tbl_name
        SET col_name=expression, col_name=expression,...

`REPLACE' works exactly like `INSERT', except that if an old record in
the table has the same value as a new record on a unique index, the old
record is deleted before the new record is inserted.
__replace__

use constant H_LOAD_DATA => <<'__load_data__';
LOAD DATA [LOW_PRIORITY] [LOCAL] INFILE 'file_name.txt' [REPLACE | IGNORE]
    INTO TABLE tbl_name
    [FIELDS
        [TERMINATED BY '\t']
        [OPTIONALLY] ENCLOSED BY '']
        [ESCAPED BY '\\' ]]
    [LINES TERMINATED BY '\n']
    [IGNORE number LINES]
    [(col_name,...)]

The `LOAD DATA INFILE' statement reads rows from a text file into a
table at a very high speed.  If the `LOCAL' keyword is specified, the
file is read from the client host.  If `LOCAL' is not specified, the
file must be located on the server.  (`LOCAL' is available in *MySQL*
3.22.6 or later.)

For security reasons, when reading text files located on the server, the
files must either reside in the database directory or be readable by
all.  Also, to use `LOAD DATA INFILE' on server files, you must have the
*file* privilege on the server host.  *Note Privileges provided::.

If you specify the keyword `LOW_PRIORITY', execution of the `LOAD DATA'
statement is delayed until no other clients are reading from the table.

Using `LOCAL' will be a bit slower than letting the server access the
files directly, because the contents of the file must travel from the
client host to the server host.  On the other hand, you do not need the
*file* privilege to load local files.

You can also load data files by using the `mysqlimport' utility; it
operates by sending a `LOAD DATA INFILE' command to the server.  The
`--local' option causes `mysqlimport' to read data files from the
client host.  You can specify the `--compress' option to get better
performance over slow networks if the client and server support the
compressed protocol.

When locating files on the server host, the server uses the following
rules:

   * If an absolute pathname is given, the server uses the pathname as
     is.

   * If a relative pathname with one or more leading components is
     given, the server searches for the file relative to the server's
     data directory.

   * If a filename with no leading components is given, the server
     looks for the file in the database directory of the current
     database.

Note that these rules mean a file given as `./myfile.txt' is read from
the server's data directory, whereas a file given as `myfile.txt' is
read from the database directory of the current database.  For example,
the following `LOAD DATA' statement reads the file `data.txt' from the
database directory for `db1' because `db1' is the current database,
even though the statement explicitly loads the file into a table in the
`db2' database:

     mysql> USE db1;
     mysql> LOAD DATA INFILE "data.txt" INTO TABLE db2.my_table;

The `REPLACE' and `IGNORE' keywords control handling of input records
that duplicate existing records on unique key values.  If you specify
`REPLACE', new rows replace existing rows that have the same unique key
value. If you specify `IGNORE', input rows that duplicate an existing
row on a unique key value are skipped.  If you don't specify either
option, an error occurs when a duplicate key value is found, and the
rest of the text file is ignored.

If you load data from a local file using the `LOCAL' keyword, the server
has no way to stop transmission of the file in the middle of the
operation, so the default bahavior is the same as if `IGNORE' is
specified.

`LOAD DATA INFILE' is the complement of `SELECT ... INTO OUTFILE'.
*Note `SELECT': SELECT.  To write data from a database to a file, use
`SELECT ... INTO OUTFILE'.  To read the file back into the database,
use `LOAD DATA INFILE'.  The syntax of the `FIELDS' and `LINES' clauses
is the same for both commands.  Both clauses are optional, but `FIELDS'
must precede `LINES' if both are specified.

If you specify a `FIELDS' clause, each of its subclauses (`TERMINATED
BY', `[OPTIONALLY] ENCLOSED BY' and `ESCAPED BY') is also optional,
except that you must specify at least one of them.

If you don't specify a `FIELDS' clause, the defaults are the same as if
you had written this:

     FIELDS TERMINATED BY '\t' ENCLOSED BY '' ESCAPED BY '\\'

If you don't specify a `LINES' clause, the default is the same as if
you had written this:

     LINES TERMINATED BY '\n'

__load_data__

use constant H_UPDATE => <<'__update__';
UPDATE [LOW_PRIORITY] [IGNORE] tbl_name SET col_name1=expr1,col_name2=expr2,...
       [WHERE where_definition] [LIMIT #]

`UPDATE' updates columns in existing table rows with new values.  The
`SET' clause indicates which columns to modify and the values they
should be given.  The `WHERE' clause, if given, specifies which rows
should be updated.  Otherwise all rows are updated.

If you specify the keyword `LOW_PRIORITY', execution of the `UPDATE' is
delayed until no other clients are reading from the table.

If you specify the keyword `IGNORE', the update statement will not
abort even if we get duplicate key errors during the update.  Rows that
would cause conflicts will not be updated.

If you access a column from `tbl_name' in an expression, `UPDATE' uses
the current value of the column.  For example, the following statement
sets the `age' column to one more than its current value:

     mysql> UPDATE persondata SET age=age+1;

`UPDATE' assignments are evaluated from left to right.  For example, the
following statement doubles the `age' column, then increments it:

     mysql> UPDATE persondata SET age=age*2, age=age+1;

If you set a column to the value it currently has, *MySQL* notices this
and doesn't update it.

`UPDATE' returns the number of rows that were actually changed.  In
*MySQL* 3.22 or later, the C API function `mysql_info()' returns the
number of rows that were matched and updated and the number of warnings
that occurred during the `UPDATE'.

In *MySQL* 3.23 you can use `LIMIT #' to ensure that only a given
number of rows are changed.

__update__

use constant H_USE => <<'__use__';
USE db_name

The `USE db_name' statement tells *MySQL* to use the `db_name' database
as the default database for subsequent queries.  The database remains
current until the end of the session, or until another `USE' statement
is issued:

     mysql> USE db1;
     mysql> SELECT count(*) FROM mytable;      # selects from db1.mytable
     mysql> USE db2;
     mysql> SELECT count(*) FROM mytable;      # selects from db2.mytable

Making a particular database current by means of the `USE' statement
does not preclude you from accessing tables in other databases.  The
example below accesses the `author' table from the `db1' database and
the `editor' table from the `db2' database:

     mysql> USE db1;
     mysql> SELECT author_name,editor_name FROM author,db2.editor
                WHERE author.editor_id = db2.editor.editor_id;

The `USE' statement is provided for Sybase compatibility.
__use__

use constant H_FLUSH => <<'__flush__';
FLUSH flush_option [,flush_option]

You should use the `FLUSH' command if you want to clear some of the
internal caches *MySQL* uses.  To execute `FLUSH', you must have the
*reload* privilege.

`flush_option' can be any of the following:

`HOSTS'     Empties the host cache tables.  You should flush the host
            tables if some of your hosts change IP number or if you get
            the error message `Host ... is blocked'.  When more than
            `max_connect_errors' errors occur in a row for a given host
            while connection to the *MySQL* server, *MySQL* assumes
            something is wrong and blocks the host from further
            connection requests.  Flushing the host tables allows the
            host to attempt to connect again. *Note Blocked host::.) You
            can start `mysqld' with `-O max_connection_errors=999999999'
            to avoid this error message.
`LOGS'      Closes and reopens the standard and update log files.  If
            you have specified the update log file without an extension,
            the extension number of the new update log file will be
            incremented by one relative to the previous file.  If you
            have used an extension in the file name, *MySQL* will close
            and reopen the update log file.  *Note Update log::.
`PRIVILEGES'Reloads the privileges from the grant tables in the `mysql'
            database.
`TABLES'    Closes all open tables.
`TABLES     Closes all open tables and locks all tables for all
WITH READ   databases with a read until one executes `UNLOCK TABLES'.
LOCK'       
`STATUS'    Resets most status variables to zero.

You can also access each of the commands shown above with the
`mysqladmin' utility, using the `flush-hosts', `flush-logs', `reload'
or `flush-tables' commands.

__flush__

use constant H_KILL => <<'__kill__';
KILL thread_id

Each connection to `mysqld' runs in a separate thread.  You can see
which threads are running with the `SHOW PROCESSLIST' command, and kill
a thread with the `KILL thread_id' command.

If you have the *process* privilege, you can see and kill all threads.
Otherwise, you can see and kill only your own threads.

You can also use the `mysqladmin processlist' and `mysqladmin kill'
commands to examine and kill threads.
__kill__

use constant H_SHOW => <<'__show__';
SHOW DATABASES             [LIKE wild]
SHOW TABLES                [FROM db_name] [LIKE wild]
SHOW COLUMNS FROM tbl_name [FROM db_name] [LIKE wild]
SHOW INDEX   FROM tbl_name [FROM db_name]
SHOW TABLE STATUS          [FROM db_name] [LIKE wild]
SHOW STATUS                [LIKE wild]
SHOW VARIABLES             [LIKE wild]
SHOW [FULL] PROCESSLIST
SHOW GRANTS FOR user

`SHOW' provides information about databases, tables, columns or status
information about the server. If the `LIKE wild' part is used, the
`wild' string can be a string that uses the SQL `%' and `_' wildcard
characters.
__show__

use constant H_EXPLAIN => <<'__explain__';
EXPLAIN tbl_name
EXPLAIN SELECT select_options

`EXPLAIN tbl_name' is a synonym for `DESCRIBE tbl_name' or `SHOW
COLUMNS FROM tbl_name'.

When you precede a `SELECT' statement with the keyword `EXPLAIN',
*MySQL* explains how it would process the `SELECT', providing
information about how tables are joined and in which order.

With the help of `EXPLAIN', you can see when you must add indexes to
tables to get a faster `SELECT' that uses indexes to find the records.
You can also see if the optimizer joins the tables in an optimal order.
To force the optimizer to use a specific join order for a `SELECT'
statement, add a `STRAIGHT_JOIN' clause.

For non-simple joins, `EXPLAIN' returns a row of information for each
table used in the `SELECT' statement. The tables are listed in the order
they would be read.  *MySQL* resolves all joins using a single-sweep
multi-join method. This means that *MySQL* reads a row from the first
table, then finds a matching row in the second table, then in the third
table and so on. When all tables are processed, it outputs the selected
columns and backtracks through the table list until a table is found
for which there are more matching rows. The next row is read from this
table and the process continues with the next table.

Output from `EXPLAIN' includes the following columns:

`table'
     The table to which the row of output refers.

`type'
     The join type.  Information about the various types is given below.

`possible_keys'
     The `possible_keys' column indicates which indexes *MySQL* could
     use to find the rows in this table. Note that this column is
     totally independent of the order of the tables. That means that
     some of the keys in possible_keys may not be useable in practice
     with the generated table order.

     If this column is empty, there are no relevant indexes. In this
     case, you may be able to improve the performance of your query by
     examining the `WHERE' clause to see if it refers to some column or
     columns that would be suitable for indexing.  If so, create an
     appropriate index and check the query with `EXPLAIN' again. *Note
     ALTER TABLE::.

     To see what indexes a table has, use `SHOW INDEX FROM tbl_name'.

`key'
     The `key' column indicates the key that *MySQL* actually decided
     to use. The key is `NULL' if no index was chosen.  If *MySQL*
     chooses the wrong index, you can probably force *MySQL* to use
     another index by using `myisamchk --analyze', *Note myisamchk
     syntax::, or by using `USE INDEX/IGNORE INDEX'.  *Note JOIN::.

`key_len'
     The `key_len' column indicates the length of the key that *MySQL*
     decided to use.  The length is `NULL' if the `key' is `NULL'. Note
     that this tell us how many parts of a multi-part key *MySQL* will
     actually use.

`ref'
     The `ref' column shows which columns or constants are used with the
     `key' to select rows from the table.

`rows'
     The `rows' column indicates the number of rows *MySQL* believes it
     must examine to execute the query.

`Extra'
     This column contains additional information of how *MySQL* will
     resolve the query. Here follows an explanation of the different
     text strings that can be found in this column:

    `Not exists'
          *MySQL* was able to do a `LEFT JOIN' optimization on the
          query and will not examine more rows in this table for a row
          combination after it founds one rows that matches the `LEFT
          JOIN' criteria.

    ``range checked for each record (index map: #)''
          *MySQL* didn't find a real good index to use. It will instead
          for each row combination in the preceding tables do a check
          which index to use (if any) use this index to retrieve the
          rows from the table.  This isn't very fast but is of course
          faster than having to do a join without an index.

    `Using filesort'
          *MySQL* will need to do an extra pass to find out how to
          retrieve the rows in sorted order.  The sort is done by going
          through all rows according to the `join type' and storing the
          sort key + pointer to the row for all rows that match the
          `WHERE'. Then the keys are sorted. Finally the rows are
          retrieved in sorted order.

    `Using index'
          The column information is retrieved from the table using only
          information in the index tree without having to do an
          additional seek to read the actually row.  This can be done
          when all the used columns for the table are part of the same
          index.

    `Using temporary'
          To be able to resolve the query *MySQL* will need to create a
          temporary table to hold the result.  This typically happens
          if you do an `ORDER BY' on a different column set than you
          did an `GROUP BY' on.

    `Where used'
          A `WHERE' clause will be used to restrict which rows will be
          matched against the next table or sent to the client.  If you
          don't have this information and the the table is of type
          `ALL' or `index' you may have something wrong in your query
          (if you don't intend to fetch/examine all rows from the
          table).

     If you want to get your queries as fast as possible, you should
     look out for `Using filesort' and `Using temporary'.

The different join types are listed below, ordered from best to worst
type:

`system'
     The table has only one row (= system table). This is a special
     case of the `const' join type.

`const'
     The table has at most one matching row, which will be read at the
     start of the query. Because there is only one row, values from the
     column in this row can be regarded as constants by the rest of the
     optimizer. `const' tables are very fast as they are read only once!

`eq_ref'
     One row will be read from this table for each combination of rows
     from the previous tables.  This the best possible join type, other
     than the `const' types.  It is used when all parts of an index are
     used by the join and the index is `UNIQUE' or a `PRIMARY KEY'.

`ref'
     All rows with matching index values will be read from this table
     for each combination of rows from the previous tables.  `ref' is
     used if the join uses only a leftmost prefix of the key, or if the
     key is not `UNIQUE' or a `PRIMARY KEY' (in other words, if the
     join cannot select a single row based on the key value).  If the
     key that is used matches only a few rows, this join type is good.

`range'
     Only rows that are in a given range will be retrieved, using an
     index to select the rows.  The `ref' column indicates which index
     is used.

`index'
     This is the same as `ALL', except that only the index tree is
     scanned.  This is usually faster than `ALL', as the index file is
     usually smaller than the data file.

`ALL'
     A full table scan will be done for each combination of rows from
     the previous tables.  This is normally not good if the table is
     the first table not marked `const', and usually *very* bad in all
     other cases. You normally can avoid `ALL' by adding more indexes,
     so that the row can be retrieved based on constant values or
     column values from earlier tables.

You can get a good indication of how good a join is by multiplying all
values in the `rows' column of the `EXPLAIN' output. This should tell
you roughly how many rows *MySQL* must examine to execute the query.
This number is also used when you restrict queries with the
`max_join_size' variable.
__explain__

use constant H_DESCRIBE => <<'__describe__';
{DESCRIBE | DESC} tbl_name {col_name | wild}

`DESCRIBE' provides information about a table's columns.  `col_name'
may be a column name or a string containing the SQL `%' and `_'
wildcard characters.
__describe__

use constant H_LOCK => <<'__lock__';
     LOCK TABLES tbl_name [AS alias] {READ | [READ LOCAL] | [LOW_PRIORITY] WRITE}
                 [, tbl_name {READ | [LOW_PRIORITY] WRITE} ...]
     ...
     UNLOCK TABLES

`LOCK TABLES' locks tables for the current thread.  `UNLOCK TABLES'
releases any locks held by the current thread.  All tables that are
locked by the current thread are automatically unlocked when the thread
issues another `LOCK TABLES', or when the connection to the server is
closed.

If a thread obtains a `READ' lock on a table, that thread (and all other
threads) can only read from the table. If a thread obtains a `WRITE'
lock on a table, then only the thread holding the lock can `READ' from
or `WRITE' to the table.  Other threads are blocked.

The difference between `READ LOCAL' and `READ' is that `READ LOCAL'
allows non-conflicting `INSERT' statements to execute while the lock is
held.  This can't however be used if you are going to manipulate the
database files outside *MySQL* while you hold the lock.

Each thread waits (without timing out) until it obtains all the locks
it has requested.

`WRITE' locks normally have higher priority than `READ' locks, to
ensure that updates are processed as soon as possible. This means that
if one thread obtains a `READ' lock and then another thread requests a
`WRITE' lock, subsequent `READ' lock requests will wait until the
`WRITE' thread has gotten the lock and released it.  You can use
`LOW_PRIORITY WRITE' locks to allow other threads to obtain `READ'
locks while the thread is waiting for the `WRITE' lock. You should only
use `LOW_PRIORITY WRITE' locks if you are sure that there will
eventually be a time when no threads will have a `READ' lock.

When you use `LOCK TABLES', you must lock all tables that you are going
to use and you must use the same alias that you are going to use in
your queries!  If you are using a table multiple times in a query (with
aliases), you must get a lock for each alias! This policy ensures that
table locking is deadlock free.

Note that you should *NOT* lock any tables that you are using with
`INSERT DELAYED'.  This is because that in this case the `INSERT' is
done by a separate thread.

Normally, you don't have to lock tables, as all single `UPDATE'
statements are atomic; no other thread can interfere with any other
currently executing SQL statement. There are a few cases when you would
like to lock tables anyway:

__lock__

use constant H_SET => <<'__set__';
SET [OPTION] SQL_VALUE_OPTION= value, ...

`SET OPTION' sets various options that affect the operation of the
server or your client.  Any option you set remains in effect until the
current session ends, or until you set the option to a different value.

`CHARACTER SET character_set_name | DEFAULT'
     This maps all strings from and to the client with the given
     mapping.  Currently the only option for `character_set_name' is
     `cp1251_koi8', but you can easily add new mappings by editing the
     `sql/convert.cc' file in the *MySQL* source distribution.  The
     default mapping can be restored by using a `character_set_name'
     value of `DEFAULT'.

     Note that the syntax for setting the `CHARACTER SET' option differs
     from the syntax for setting the other options.

`PASSWORD = PASSWORD('some password')'
     Set the password for the current user. Any non-anonymous user can
     change his own password!

`PASSWORD FOR user = PASSWORD('some password')'
     Set the password for a specific user on the current server host.
     Only a user with access to the `mysql' database can do this.  The
     user should be given in `user@hostname' format, where `user' and
     `hostname' are exactly as they are listed in the `User' and `Host'
     columns of the `mysql.user' table entry.  For example, if you had
     an entry with `User' and `Host' fields of `'bob'' and
     `'%.loc.gov'', you would write:

          mysql> SET PASSWORD FOR bob@"%.loc.gov" = PASSWORD("newpass");
          
          or
          
          mysql> UPDATE mysql.user SET password=PASSWORD("newpass") where user="bob' and host="%.loc.gov";

`SQL_AUTO_IS_NULL = 0 | 1'
     If set to `1' (default) then one can find the last inserted row
     for a table with an auto_increment row with the following
     construct: `WHERE auto_increment_column IS NULL'.  This is used by
     some ODBC programs like Access.

`SET AUTOCOMMIT= 0 | 1'
     If set to `1' all changes to a table will be done at once. To start
     an multi command transaction you have to use the `BEGIN'
     statement. *Note COMMIT::. If set to `0' you have to use `COMMIT' /
     `ROLLBACK' to accept/revoke that transaction. *Note COMMIT::.  Note
     that when you change from not `AUTOCOMMIT' mode to `AUTOCOMMIT'
     mode, *MySQL* will do an automatic `COMMIT' on any open
     transactions.

`SQL_BIG_TABLES = 0 | 1'
     If set to `1', all temporary tables are stored on disk rather than
     in memory.  This will be a little slower, but you will not get the
     error `The table tbl_name is full' for big `SELECT' operations that
     require a large temporary table.  The default value for a new
     connection is `0' (i.e., use in-memory temporary tables).

`SQL_BIG_SELECTS = 0 | 1'
     If set to `0', *MySQL* will abort if a `SELECT' is attempted that
     probably will take a very long time. This is useful when an
     inadvisable `WHERE' statement has been issued. A big query is
     defined as a `SELECT' that probably will have to examine more than
     `max_join_size' rows.  The default value for a new connection is
     `1' (which will allow all `SELECT' statements).

`SQL_BUFFER_RESULT = 0 | 1'
     `SQL_BUFFER_RESULT' will force the result from `SELECT''s to be
     put into a temporary table. This will help *MySQL* free the table
     locks early and will help in cases where it takes a long time to
     send the result set to the client.

`SQL_LOW_PRIORITY_UPDATES = 0 | 1'
     If set to `1', all `INSERT', `UPDATE', `DELETE' and and `LOCK
     TABLE WRITE' statements wait until there is no pending `SELECT' or
     `LOCK TABLE READ' on the affected table.

`SQL_MAX_JOIN_SIZE = value | DEFAULT'
     Don't allow `SELECT''s that will probably need to examine more than
     `value' row combinations.  By setting this value, you can catch
     `SELECT''s where keys are not used properly and that would probably
     take a long time. Setting this to a value other than `DEFAULT'
     will reset the `SQL_BIG_SELECTS' flag.  If you set the
     `SQL_BIG_SELECTS' flag again, the `SQL_MAX_JOIN_SIZE' variable
     will be ignored.  You can set a default value for this variable by
     starting `mysqld' with `-O max_join_size=#'.

`SQL_SAFE_MODE = 0 | 1'
     If set to `1', *MySQL* will abort if a `UPDATE' or `DELETE' is
     attempted that doesn't use a key or `LIMIT' in the `WHERE' clause.
     This makes it possible to catch wrong updates when creating SQL
     commands by hand.

`SQL_SELECT_LIMIT = value | DEFAULT'
     The maximum number of records to return from `SELECT' statements.
     If a `SELECT' has a `LIMIT' clause, the `LIMIT' takes precedence
     over the value of `SQL_SELECT_LIMIT'.  The default value for a new
     connection is "unlimited".  If you have changed the limit, the
     default value can be restored by using a `SQL_SELECT_LIMIT' value
     of `DEFAULT'.

`SQL_LOG_OFF = 0 | 1'
     If set to `1', no logging will be done to the standard log for this
     client, if the client has the *process* privilege.  This does not
     affect the update log!

`SQL_LOG_UPDATE = 0 | 1'
     If set to `0', no logging will be done to the update log for the
     client, if the client has the *process* privilege.  This does not
     affect the standard log!

`TIMESTAMP = timestamp_value | DEFAULT'
     Set the time for this client.  This is used to get the original
     timestamp if you use the update log to restore rows.

`LAST_INSERT_ID = #'
     Set the value to be returned from `LAST_INSERT_ID()'. This is
     stored in the update log when you use `LAST_INSERT_ID()' in a
     command that updates a table.

`INSERT_ID = #'
     Set the value to be used by the following `INSERT' command when
     inserting an `AUTO_INCREMENT' value.  This is mainly used with the
     update log.
__set__

use constant H_GRANT => <<'__grant__';
GRANT priv_type [(column_list)] [, priv_type [(column_list)] ...]
         ON {tbl_name | * | *.* | db_name.*}
         TO user_name [IDENTIFIED BY 'password']
             [, user_name [IDENTIFIED BY 'password'] ...]
         [WITH GRANT OPTION]
     
     REVOKE priv_type [(column_list)] [, priv_type [(column_list)] ...]
         ON {tbl_name | * | *.* | db_name.*}
         FROM user_name [, user_name ...]

`GRANT' is implemented in *MySQL* 3.22.11 or later. For earlier *MySQL*
versions, the `GRANT' statement does nothing.

The `GRANT' and `REVOKE' commands allow system administrators to grant
and revoke rights to *MySQL* users at four privilege levels:

*Global level*
     Global privileges apply to all databases on a given server. These
     privileges are stored in the `mysql.user' table.

*Database level*
     Database privileges apply to all tables in a given database. These
     privileges are stored in the `mysql.db' and `mysql.host' tables.

*Table level*
     Table privileges apply to all columns in a given table. These
     privileges are stored in the `mysql.tables_priv' table.

*Column level*
     Column privileges apply to single columns in a given table. These
     privileges are stored in the `mysql.columns_priv' table.

For examples of how `GRANT' works, see *Note Adding users::.

For the `GRANT' and `REVOKE' statements, `priv_type' may be specified
as any of the following:

     ALL PRIVILEGES      FILE                RELOAD
     ALTER               INDEX               SELECT
     CREATE              INSERT              SHUTDOWN
     DELETE              PROCESS             UPDATE
     DROP                REFERENCES          USAGE

`ALL' is a synonym for `ALL PRIVILEGES'.  `REFERENCES' is not yet
implemented.  `USAGE' is currently a synonym for "no privileges".  It
can be used when you want to create a user that has no privileges.

To revoke the *grant* privilege from a user, use a `priv_type' value of
`GRANT OPTION':

     REVOKE GRANT OPTION ON ... FROM ...;

The only `priv_type' values you can specify for a table are `SELECT',
`INSERT', `UPDATE', `DELETE', `CREATE', `DROP', `GRANT', `INDEX' and
`ALTER'.

The only `priv_type' values you can specify for a column (that is, when
you use a `column_list' clause) are `SELECT', `INSERT' and `UPDATE'.

You can set global privileges by using `ON *.*' syntax.  You can set
database privileges by using `ON db_name.*' syntax. If you specify `ON
*' and you have a current database, you will set the privileges for
that database.  (*Warning:* If you specify `ON *' and you _don't_ have
a current database, you will affect the global privileges!)

__grant__

use constant H_CREATE_INDEX => <<'__create_index__';
CREATE [UNIQUE] INDEX index_name ON tbl_name (col_name[(length)],... )

The `CREATE INDEX' statement doesn't do anything in *MySQL* prior to
version 3.22.  In 3.22 or later, `CREATE INDEX' is mapped to an `ALTER
TABLE' statement to create indexes.  *Note `ALTER TABLE': ALTER TABLE.

Normally, you create all indexes on a table at the time the table itself
is created with `CREATE TABLE'.  *Note `CREATE TABLE': CREATE TABLE.
`CREATE INDEX' allows you to add indexes to existing tables.

A column list of the form `(col1,col2,...)' creates a multiple-column
index.  Index values are formed by concatenating the values of the given
columns.

For `CHAR' and `VARCHAR' columns, indexes can be created that use only
part of a column, using `col_name(length)' syntax.  (On `BLOB' and
`TEXT' columns the length is required). The statement shown below
creates an index using the first 10 characters of the `name' column:

     mysql> CREATE INDEX part_of_name ON customer (name(10));

__create_index__

use constant H_DROP_INDEX => <<'__drop_index__';
DROP INDEX index_name ON tbl_name

`DROP INDEX' drops the index named `index_name' from the table
`tbl_name'.  `DROP INDEX' doesn't do anything in *MySQL* prior to
version 3.22.  In 3.22 or later, `DROP INDEX' is mapped to an `ALTER
TABLE' statement to drop the index.  *Note `ALTER TABLE': ALTER TABLE.
__drop_index__

use constant H_COMMENTS => <<'__comments__';
The *MySQL* server supports the `# to end of line', `-- to end of line'
and `/* in-line or multiple-line */' comment styles:

     mysql> select 1+1;     # This comment continues to the end of line
     mysql> select 1+1;     -- This comment continues to the end of line
     mysql> select 1 /* this is an in-line comment */ + 1;
     mysql> select 1+
     /*
     this is a
     multiple-line comment
     */
     1;

Note that the `--' comment style requires you to have at least one space
after the `--'!

Although the server understands the comment syntax just described,
there are some limitations on the way that the `mysql' client parses
`/* ... */' comments:

   * Single-quote and double-quote characters are taken to indicate the
     beginning of a quoted string, even within a comment.  If the quote
     is not matched by a second quote within the comment, the parser
     doesn't realize the comment has ended.  If you are running `mysql'
     interactively, you can tell that it has gotten confused like this
     because the prompt changes from `mysql>' to `'>' or `">'.

   * A semicolon is taken to indicate the end of the current SQL
     statement and anything following it to indicate the beginning of
     the next statement.

These limitations apply both when you run `mysql' interactively and
when you put commands in a file and tell `mysql' to read its input from
that file with `mysql < some-file'.
__comments__

use constant H_CREATE_FUNCTION => <<'__create_function__';
CREATE [AGGREGATE] FUNCTION function_name RETURNS {STRING|REAL|INTEGER}
            SONAME shared_library_name
     
     DROP FUNCTION function_name

A user-definable function (UDF) is a way to extend *MySQL* with a new
function that works like native (built in) *MySQL* functions such as
`ABS()' and `CONCAT()'.

`AGGREGATE' is a new option for *MySQL* 3.23.  An `AGGREGATE' function
works exactly like a native *MySQL* `GROUP' function like `SUM' or
`COUNT()'.

`CREATE FUNCTION' saves the function's name, type and shared library
name in the `mysql.func' system table.  You must have the *insert* and
*delete* privileges for the `mysql' database to create and drop
functions.
__create_function__

use constant H_RESERVED_WORDS => <<'__reserved_words__';
Is MySQL picky about reserved words?
====================================

A common problem stems from trying to create a table with column names
that use the names of datatypes or functions built into *MySQL*, such as
`TIMESTAMP' or `GROUP'.  You're allowed to do it (for example, `ABS' is
an allowed column name), but whitespace is not allowed between a
function name and the `(' when using functions whose names are also
column names.

The following words are explicitly reserved in *MySQL*. Most of them
are forbidden by ANSI SQL92 as column and/or table names (for example,
`group').  A few are reserved because *MySQL* needs them and is
(currently) using a `yacc' parser:

`action'           `add'              `aggregate'        `all'
`alter'            `after'            `and'              `as'
`asc'              `avg'              `avg_row_length'   `auto_increment'
`between'          `bigint'           `bit'              `binary'
`blob'             `bool'             `both'             `by'
`cascade'          `case'             `char'             `character'
`change'           `check'            `checksum'         `column'
`columns'          `comment'          `constraint'       `create'
`cross'            `current_date'     `current_time'     `current_timestamp'
`data'             `database'         `databases'        `date'
`datetime'         `day'              `day_hour'         `day_minute'
`day_second'       `dayofmonth'       `dayofweek'        `dayofyear'
`dec'              `decimal'          `default'          `delayed'
`delay_key_write'  `delete'           `desc'             `describe'
`distinct'         `distinctrow'      `double'           `drop'
`end'              `else'             `escape'           `escaped'
`enclosed'         `enum'             `explain'          `exists'
`fields'           `file'             `first'            `float'
`float4'           `float8'           `flush'            `foreign'
`from'             `for'              `full'             `function'
`global'           `grant'            `grants'           `group'
`having'           `heap'             `high_priority'    `hour'
`hour_minute'      `hour_second'      `hosts'            `identified'
`ignore'           `in'               `index'            `infile'
`inner'            `insert'           `insert_id'        `int'
`integer'          `interval'         `int1'             `int2'
`int3'             `int4'             `int8'             `into'
`if'               `is'               `isam'             `join'
`key'              `keys'             `kill'             `last_insert_id'
`leading'          `left'             `length'           `like'
`lines'            `limit'            `load'             `local'
`lock'             `logs'             `long'             `longblob'
`longtext'         `low_priority'     `max'              `max_rows'
`match'            `mediumblob'       `mediumtext'       `mediumint'
`middleint'        `min_rows'         `minute'           `minute_second'
`modify'           `month'            `monthname'        `myisam'
`natural'          `numeric'          `no'               `not'
`null'             `on'               `optimize'         `option'
`optionally'       `or'               `order'            `outer'
`outfile'          `pack_keys'        `partial'          `password'
`precision'        `primary'          `procedure'        `process'
`processlist'      `privileges'       `read'             `real'
`references'       `reload'           `regexp'           `rename'
`replace'          `restrict'         `returns'          `revoke'
`rlike'            `row'              `rows'             `second'
`select'           `set'              `show'             `shutdown'
`smallint'         `soname'           `sql_big_tables'   `sql_big_selects'
`sql_low_priority_updates'`sql_log_off'      `sql_log_update'   `sql_select_limit'
`sql_small_result' `sql_big_result'   `sql_warnings'     `straight_join'
`starting'         `status'           `string'           `table'
`tables'           `temporary'        `terminated'       `text'
`then'             `time'             `timestamp'        `tinyblob'
`tinytext'         `tinyint'          `trailing'         `to'
`type'             `use'              `using'            `unique'
`unlock'           `unsigned'         `update'           `usage'
`values'           `varchar'          `variables'        `varying'
`varbinary'        `with'             `write'            `when'
`where'            `year'             `year_month'       `zerofill'

The following symbols (from the table above) are disallowed by ANSI SQL
but allowed by *MySQL* as column/table names. This is because some of
these names are very natural names and a lot of people have already
used them.

   * `ACTION'

   * `BIT'

   * `DATE'

   * `ENUM'

   * `NO'

   * `TEXT'

   * `TIME'

   * `TIMESTAMP'
__reserved_words__

%HELP =
  (
   arithmetic           => H_ARITHMETIC,
   bit_operators        => H_BIT_OPERATORS,
   logical_operators    => H_LOGICAL_OPERATORS,
   comparison           => H_COMPARISON,
   isnull               => H_ISNULL,
   coalesce             => H_COALESCE,
   interval             => H_INTERVAL,
   string_comparison    => H_STRING_COMPARISON,
   strcmp               => H_STRCMP,
   binary               => H_BINARY,
   ifnull               => H_IFNULL,
   nullif               => H_NULLIF,
   if                   => H_IF,
   case                 => H_CASE,
   flow_control         => H_FLOW_CONTROL,
   abs                  => H_ABS,
   sign                 => H_SIGN,
   mod                  => H_MOD,
   floor                => H_FLOOR,
   ceiling              => H_CEILING,
   round                => H_ROUND,
   round                => H_ROUND,
   exp                  => H_EXP,
   log                  => H_LOG,
   log10                => H_LOG10,
   pow                  => H_POW,
   sqrt                 => H_SQRT,
   pi                   => H_PI,
   cos                  => H_COS,
   sin                  => H_SIN,
   tan                  => H_TAN,
   acos                 => H_ACOS,
   asin                 => H_ASIN,
   atan                 => H_ATAN,
   atan2                => H_ATAN2,
   cot                  => H_COT,
   rand                 => H_RAND,
   least                => H_LEAST,
   greatest             => H_GREATEST,
   degrees              => H_DEGREES,
   radians              => H_RADIANS,
   truncate             => H_TRUNCATE,
   numeric_functions    => H_NUMERIC_FUNCTIONS,
   ascii                => H_ASCII,
   ord                  => H_ORD,
   conv                 => H_CONV,
   bin                  => H_BIN,
   oct                  => H_OCT,
   hex                  => H_HEX,
   char                 => H_CHAR,
   concat               => H_CONCAT,
   concat_ws            => H_CONCAT_WS,
   length               => H_LENGTH,
   octet_length         => H_OCTET_LENGTH,
   char_length          => H_CHAR_LENGTH,
   character_length     => H_CHARACTER_LENGTH,
   locate               => H_LOCATE,
   position             => H_POSITION,
   instr                => H_INSTR,
   lpad                 => H_LPAD,
   rpad                 => H_RPAD,
   left                 => H_LEFT,
   right                => H_RIGHT,
   mid                  => H_MID,
   substring            => H_SUBSTRING,
   substring_index      => H_SUBSTRING_INDEX,
   ltrim                => H_LTRIM,
   rtrim                => H_RTRIM,
   trim                 => H_TRIM,
   soundex              => H_SOUNDEX,
   space                => H_SPACE,
   fn_replace           => H_FN_REPLACE,
   repeat               => H_REPEAT,
   reverse              => H_REVERSE,
   fn_insert            => H_FN_INSERT,
   elt                  => H_ELT,
   field                => H_FIELD,
   find_in_set          => H_FIND_IN_SET,
   make_set             => H_MAKE_SET,
   export_set           => H_EXPORT_SET,
   lcase                => H_LCASE,
   lower                => H_LOWER,
   ucase                => H_UCASE,
   upper                => H_UPPER,
   load_file            => H_LOAD_FILE,
   string_functions     => H_STRING_FUNCTIONS,
   dayofweek            => H_DAYOFWEEK,
   weekday              => H_WEEKDAY,
   dayofmonth           => H_DAYOFMONTH,
   dayofyear            => H_DAYOFYEAR,
   month                => H_MONTH,
   dayname              => H_DAYNAME,
   monthname            => H_MONTHNAME,
   quarter              => H_QUARTER,
   week                 => H_WEEK,
   year                 => H_YEAR,
   yearweek             => H_YEARWEEK,
   hour                 => H_HOUR,
   minute               => H_MINUTE,
   second               => H_SECOND,
   period_add           => H_PERIOD_ADD,
   period_diff          => H_PERIOD_DIFF,
   date_add             => H_DATE_ADD,
   to_days              => H_TO_DAYS,
   from_days            => H_FROM_DAYS,
   date_format          => H_DATE_FORMAT,
   time_format          => H_TIME_FORMAT,
   curdate              => H_CURDATE,
   curtime              => H_CURTIME,
   now                  => H_NOW,
   unix_timestamp       => H_UNIX_TIMESTAMP,
   from_unixtime        => H_FROM_UNIXTIME,
   sec_to_time          => H_SEC_TO_TIME,
   time_to_sec          => H_TIME_TO_SEC,
   date_time_functions  => H_DATE_TIME_FUNCTIONS,
   database             => H_DATABASE,
   user                 => H_USER,
   password             => H_PASSWORD,
   encrypt              => H_ENCRYPT,
   encode               => H_ENCODE,
   decode               => H_DECODE,
   md5                  => H_MD5,
   last_insert_id       => H_LAST_INSERT_ID,
   format               => H_FORMAT,
   version              => H_VERSION,
   connection_id        => H_CONNECTION_ID,
   get_lock             => H_GET_LOCK,
   release_lock         => H_RELEASE_LOCK,
   benchmark            => H_BENCHMARK,
   inet_ntoa            => H_INET_NTOA,
   inet_aton            => H_INET_ATON,
   misc_functions       => H_MISC_FUNCTIONS,
   count                => H_COUNT,
   avg                  => H_AVG,
   min                  => H_MIN,
   sum                  => H_SUM,
   std                  => H_STD,
   bit_or               => H_BIT_OR,
   bit_and              => H_BIT_AND,
   group_by_functions   => H_GROUP_BY_FUNCTIONS,
   data_types           => H_DATA_TYPES,
   mysql_variables      => H_MYSQL_VARIABLES,
   create_database      => H_CREATE_DATABASE,
   drop_database        => H_DROP_DATABASE,
   alter_table          => H_ALTER_TABLE,
   drop_table           => H_DROP_TABLE,
   optimize_table       => H_OPTIMIZE_TABLE,
   check_table          => H_CHECK_TABLE,
   repair_table         => H_REPAIR_TABLE,
   delete               => H_DELETE,
   select               => H_SELECT,
   join                 => H_JOIN,
   insert               => H_INSERT,
   replace              => H_REPLACE,
   load_data            => H_LOAD_DATA,
   update               => H_UPDATE,
   use                  => H_USE,
   flush                => H_FLUSH,
   kill                 => H_KILL,
   show                 => H_SHOW,
   explain              => H_EXPLAIN,
   describe             => H_DESCRIBE,
   lock                 => H_LOCK,
   set                  => H_SET,
   grant                => H_GRANT,
   create_index         => H_CREATE_INDEX,
   drop_index           => H_DROP_INDEX,
   comments             => H_COMMENTS,
   create_function      => H_CREATE_FUNCTION,
   reserved_words       => H_RESERVED_WORDS,
   power                => H_POW,
   date_sub             => H_DATE_ADD,
   adddate              => H_DATE_ADD,
   subdate              => H_DATE_ADD,
   sysdate              => H_NOW,
   system_user          => H_USER,
   session_user         => H_USER,
   max                  => H_MIN,
   stddev               => H_STD   
  );

sub help_map ($) { return \%HELP }


__END__


perl -n -e  'if(/^\`(\w+)\((.*)\)\x27/){ $lim && print(STDOUT "__${lim}__\n\n"); print(STDOUT "use constant H_$1 => <<\x27__",lc($1),"__\x27;\n$1($2):\n\n"); $lim = lc($1) } else { print(STDOUT $_) } END { print(STDOUT "__${lim}__\n") }'




